<?php
declare(strict_types=1);

function e(string $value): string {
    return htmlspecialchars($value, ENT_QUOTES, 'UTF-8');
}

function base_path(): string {
    static $base = null;
    if ($base !== null) {
        return $base;
    }

    $configured = '';
    if (defined('APP_BASE_PATH')) {
        $configured = trim((string)APP_BASE_PATH);
    }
    if ($configured !== '') {
        $configured = '/' . trim($configured, '/');
        $base = ($configured === '/') ? '' : $configured;
        return $base;
    }

    $script = (string)($_SERVER['SCRIPT_NAME'] ?? '');
    $candidate = '';

    $pos = strpos($script, '/public/');
    if ($pos !== false) {
        $candidate = substr($script, 0, $pos + 7); // inclui /public
    } elseif (preg_match('#^(.*?/public)$#', $script, $m)) {
        $candidate = $m[1];
    }

    $candidate = rtrim($candidate, '/');
    $base = ($candidate === '' || $candidate === '/') ? '' : $candidate;
    return $base;
}

function url(string $path = '/'): string {
    $path = trim($path);
    if ($path === '') $path = '/';

    if (preg_match('#^(?:https?:)?//#i', $path) || str_starts_with($path, 'mailto:') || str_starts_with($path, 'tel:') || str_starts_with($path, '#')) {
        return $path;
    }

    $base = base_path();
    if ($path === '/') {
        return ($base !== '' ? $base : '') . '/';
    }
    return ($base !== '' ? $base : '') . '/' . ltrim($path, '/');
}

function app_url(string $path = '/'): string {
    $root = rtrim((string)APP_URL, '/');
    $parsedPath = (string)(parse_url($root, PHP_URL_PATH) ?? '');
    if (($parsedPath === '' || $parsedPath === '/') && base_path() !== '') {
        $root .= base_path();
    }
    if ($path === '' || $path === '/') {
        return rtrim($root, '/') . '/';
    }
    if (preg_match('#^https?://#i', $path)) {
        return $path;
    }
    return rtrim($root, '/') . '/' . ltrim($path, '/');
}

function redirect(string $path): never {
    header('Location: ' . url($path));
    exit;
}

function is_post(): bool {
    return strtoupper($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'POST';
}

function csrf_token(): string {
    if (empty($_SESSION['_csrf'])) {
        $_SESSION['_csrf'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['_csrf'];
}

function csrf_check(?string $token): bool {
    return is_string($token) && hash_equals($_SESSION['_csrf'] ?? '', $token);
}

function flash(string $type, string $message): void {
    $_SESSION['_flash'] = compact('type', 'message');
}

function flash_show(): ?array {
    $f = $_SESSION['_flash'] ?? null;
    unset($_SESSION['_flash']);
    return $f;
}

function auth_user(): ?array {
    return $_SESSION['user'] ?? null;
}

function refresh_auth_user(): void {
    $u = auth_user();
    if (!$u) return;
    $stmt = db()->prepare("SELECT id,name,email,role,status,current_plan_id FROM users WHERE id=:id LIMIT 1");
    $stmt->execute(['id' => (int)$u['id']]);
    $dbu = $stmt->fetch();
    if ($dbu) {
        $_SESSION['user'] = [
            'id' => (int)$dbu['id'],
            'name' => $dbu['name'],
            'email' => $dbu['email'],
            'role' => $dbu['role'],
            'status' => $dbu['status'],
            'current_plan_id' => $dbu['current_plan_id']
        ];
    }
}

function require_auth(): void {
    if (!auth_user()) {
        flash('warning', 'Faça login para continuar.');
        redirect('/auth/login.php');
    }
}

function require_admin(): void {
    require_auth();
    if ((auth_user()['role'] ?? '') !== 'admin') {
        http_response_code(403);
        echo 'Acesso negado';
        exit;
    }
}

function app_path(string $path = ''): string {
    return dirname(__DIR__) . ($path ? '/' . ltrim($path, '/') : '');
}

function storage_path(string $path = ''): string {
    return dirname(__DIR__) . '/storage' . ($path ? '/' . ltrim($path, '/') : '');
}

function ensure_upload_dir(): string {
    // Para facilitar em hospedagens compartilhadas, os uploads ficam em /public/uploads
    $p = dirname(__DIR__) . '/public/uploads';
    if (!is_dir($p)) {
        mkdir($p, 0775, true);
    }
    return $p;
}

function ensure_hls_video_dir(int $videoId): string {
    $p = storage_path('hls/' . $videoId);
    if (!is_dir($p)) {
        mkdir($p, 0775, true);
    }
    return $p;
}

function human_interval(string $unit, int $count): string {
    $dict = ['day' => 'dia', 'week' => 'semana', 'month' => 'mês', 'year' => 'ano'];
    $u = $dict[$unit] ?? $unit;
    return $count . ' ' . $u . ($count > 1 ? 's' : '');
}

function bytes_to_mb(int $bytes): float {
    return round($bytes / (1024 * 1024), 2);
}

function human_bytes(int $bytes): string {
    if ($bytes < 1024) return $bytes . ' B';
    $units = ['KB','MB','GB','TB'];
    $v = $bytes;
    $i = -1;
    while ($v >= 1024 && $i < count($units)-1) {
        $v /= 1024;
        $i++;
    }
    return number_format($v, 2, ',', '.') . ' ' . $units[$i];
}

function now_ym(): string {
    return date('Y-m');
}

function month_bounds(string $ym): array {
    $start = $ym . '-01 00:00:00';
    $dt = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', $start);
    $end = $dt ? $dt->modify('+1 month')->format('Y-m-d H:i:s') : date('Y-m-01 00:00:00', strtotime('+1 month'));
    return [$start, $end];
}
function ffmpeg_binary_info(string $bin): array {
    $bin = trim($bin);
    if ($bin === '') {
        return ['ok' => false, 'bin' => '', 'resolved' => null, 'message' => 'Binário não configurado'];
    }

    // Caminho absoluto/relativo com separador ou .exe explícito
    $looksLikePath = str_contains($bin, '/') || str_contains($bin, '\\') || str_ends_with(strtolower($bin), '.exe');
    if ($looksLikePath) {
        $resolved = $bin;
        if (file_exists($resolved)) {
            return ['ok' => true, 'bin' => $bin, 'resolved' => $resolved, 'message' => 'Encontrado'];
        }
        return ['ok' => false, 'bin' => $bin, 'resolved' => $resolved, 'message' => 'Caminho não encontrado'];
    }

    // Resolver no PATH
    $cmd = (PHP_OS_FAMILY === 'Windows')
        ? 'where ' . escapeshellarg($bin) . ' 2>NUL'
        : 'command -v ' . escapeshellarg($bin) . ' 2>/dev/null';
    $out = shell_exec($cmd);
    $resolved = null;
    if (is_string($out)) {
        $line = trim(strtok($out, "\n"));
        if ($line !== '') {
            $resolved = $line;
        }
    }

    if ($resolved && file_exists($resolved)) {
        return ['ok' => true, 'bin' => $bin, 'resolved' => $resolved, 'message' => 'Encontrado no PATH'];
    }

    return ['ok' => false, 'bin' => $bin, 'resolved' => null, 'message' => 'Não encontrado no PATH'];
}

function ffmpeg_health(): array {
    $ffmpeg = ffmpeg_binary_info((string)(defined('FFMPEG_BIN') ? FFMPEG_BIN : 'ffmpeg'));
    $ffprobe = ffmpeg_binary_info((string)(defined('FFPROBE_BIN') ? FFPROBE_BIN : 'ffprobe'));
    $ok = $ffmpeg['ok'] && $ffprobe['ok'];
    $message = $ok
        ? 'FFmpeg e FFprobe disponíveis.'
        : 'FFmpeg/FFprobe ausentes. Configure FFMPEG_BIN e FFPROBE_BIN no .env (ex.: C:\\ffmpeg\\bin\\ffmpeg.exe).';
    return [
        'ok' => $ok,
        'ffmpeg' => $ffmpeg,
        'ffprobe' => $ffprobe,
        'message' => $message
    ];
}
