<?php
declare(strict_types=1);

class HttpClient
{
    public static function request(string $method, string $url, array $headers = [], ?array $body = null, int $timeout = 25): array
    {
        // Prefer Guzzle when available (installed via Composer), fallback to cURL nativo
        if (class_exists('GuzzleHttp\\Client')) {
            try {
                $client = new \GuzzleHttp\Client([
                    'timeout' => $timeout,
                    'http_errors' => false,
                ]);

                $assocHeaders = [];
                foreach ($headers as $h) {
                    if (strpos($h, ':') !== false) {
                        [$k, $v] = array_map('trim', explode(':', $h, 2));
                        $assocHeaders[$k] = $v;
                    }
                }
                if (!isset($assocHeaders['Content-Type'])) {
                    $assocHeaders['Content-Type'] = 'application/json';
                }
                if (!isset($assocHeaders['Accept'])) {
                    $assocHeaders['Accept'] = 'application/json';
                }

                $options = ['headers' => $assocHeaders];
                if ($body !== null) {
                    $options['json'] = $body;
                }

                $resp = $client->request(strtoupper($method), $url, $options);
                $status = (int)$resp->getStatusCode();
                $raw = (string)$resp->getBody();
                $json = $raw !== '' ? json_decode($raw, true) : null;

                return [
                    'ok' => $status >= 200 && $status < 300,
                    'status' => $status,
                    'error' => null,
                    'body_raw' => $raw,
                    'body' => $json
                ];
            } catch (Throwable $e) {
                return [
                    'ok' => false,
                    'status' => 0,
                    'error' => $e->getMessage(),
                    'body_raw' => null,
                    'body' => null
                ];
            }
        }

        // Fallback cURL
        $ch = curl_init($url);
        $defaultHeaders = ['Content-Type: application/json', 'Accept: application/json'];
        $allHeaders = array_merge($defaultHeaders, $headers);

        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST  => strtoupper($method),
            CURLOPT_HTTPHEADER     => $allHeaders,
            CURLOPT_TIMEOUT        => $timeout,
        ]);

        if ($body !== null) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($body, JSON_UNESCAPED_UNICODE));
        }

        $response = curl_exec($ch);
        $err = curl_error($ch);
        $status = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $json = null;
        if (is_string($response) && $response !== '') {
            $json = json_decode($response, true);
        }

        return [
            'ok' => $err === '' && $status >= 200 && $status < 300,
            'status' => $status,
            'error' => $err ?: null,
            'body_raw' => $response,
            'body' => $json
        ];
    }
}
