<?php
declare(strict_types=1);
require_once __DIR__ . '/HttpClient.php';
require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';

class PagBankService
{
    private string $base;
    private string $token;

    public function __construct(?string $token = null)
    {
        $cfg = $this->readProviderSettings();
        $this->base = rtrim((string)($cfg['api_base'] ?? PB_API_BASE), '/');
        $this->token = (string)($token ?: ($cfg['access_token'] ?? PB_ACCESS_TOKEN));
    }

    private function readProviderSettings(): array
    {
        try {
            $stmt = db()->prepare("SELECT * FROM payment_settings WHERE provider='pagbank' LIMIT 1");
            $stmt->execute();
            $s = $stmt->fetch();
            if (!$s || (int)$s['is_enabled'] !== 1) return [];
            $extra = [];
            if (!empty($s['extra_json'])) {
                $tmp = json_decode((string)$s['extra_json'], true);
                if (is_array($tmp)) $extra = $tmp;
            }
            return array_merge($s, $extra);
        } catch (Throwable $e) {
            return [];
        }
    }

    private function authHeader(): array
    {
        return ["Authorization: Bearer {$this->token}"];
    }

    public function createPlan(array $payload): array
    {
        return HttpClient::request('POST', "{$this->base}/plans", $this->authHeader(), $payload);
    }

    public function createCustomer(array $payload): array
    {
        return HttpClient::request('POST', "{$this->base}/customers", $this->authHeader(), $payload);
    }

    public function createSubscription(array $payload): array
    {
        return HttpClient::request('POST', "{$this->base}/subscriptions", $this->authHeader(), $payload);
    }

    public function getSubscription(string $id): array
    {
        return HttpClient::request('GET', "{$this->base}/subscriptions/" . urlencode($id), $this->authHeader());
    }

    public function activateSubscription(string $id): array
    {
        return HttpClient::request('PUT', "{$this->base}/subscriptions/" . urlencode($id) . '/activate', $this->authHeader());
    }

    public function verifyNotification(string $notificationCode): array
    {
        $url = PB_WEBHOOK_VALIDATE_ENDPOINT . '?notificationCode=' . urlencode($notificationCode);
        return HttpClient::request('GET', $url, $this->authHeader());
    }
}
