<?php
declare(strict_types=1);
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../helpers.php';

class PlanGuardService
{
    public static function resolveActivePlan(int $userId): ?array
    {
        // 1) current_plan_id explícito no usuário
        $stmt = db()->prepare("
            SELECT p.*
            FROM users u
            JOIN plans p ON p.id = u.current_plan_id
            WHERE u.id=:u AND p.is_enabled=1
            LIMIT 1
        ");
        $stmt->execute(['u' => $userId]);
        $p = $stmt->fetch();
        if ($p) return $p;

        // 2) último subscription ativo/autorizado
        $stmt = db()->prepare("
            SELECT p.*
            FROM subscriptions s
            JOIN plans p ON p.id = s.plan_id
            WHERE s.user_id=:u
              AND s.status IN ('active','authorized')
              AND p.is_enabled=1
            ORDER BY s.id DESC
            LIMIT 1
        ");
        $stmt->execute(['u' => $userId]);
        $p = $stmt->fetch();
        if ($p) {
            db()->prepare("UPDATE users SET current_plan_id=:p WHERE id=:u")->execute(['p' => $p['id'], 'u' => $userId]);
            return $p;
        }

        return null;
    }

    public static function getUsage(int $userId, ?string $ym = null): array
    {
        $ym = $ym ?: now_ym();
        [$start, $end] = month_bounds($ym);

        // armazenamento local efetivamente ocupado (vídeos em processamento/prontos)
        $q1 = db()->prepare("
            SELECT COALESCE(SUM(file_size_bytes),0) c
            FROM videos
            WHERE user_id=:u
              AND storage_type='local'
              AND status IN ('processing','ready')
        ");
        $q1->execute(['u' => $userId]);
        $storage = (int)($q1->fetch()['c'] ?? 0);

        $q2 = db()->prepare("
            SELECT COALESCE(SUM(bytes_sent),0) c
            FROM bandwidth_logs
            WHERE user_id=:u
              AND created_at >= :s
              AND created_at < :e
        ");
        $q2->execute(['u' => $userId, 's' => $start, 'e' => $end]);
        $bandwidth = (int)($q2->fetch()['c'] ?? 0);

        return [
            'period_ym' => $ym,
            'storage_used_bytes' => $storage,
            'bandwidth_used_bytes' => $bandwidth,
        ];
    }

    public static function canUpload(int $userId, int $newFileBytes): array
    {
        $plan = self::resolveActivePlan($userId);
        if (!$plan) {
            return [
                'ok' => false,
                'code' => 'no_plan',
                'message' => 'Você precisa de um plano ativo para hospedar vídeo local.',
                'plan' => null,
                'usage' => self::getUsage($userId)
            ];
        }

        $usage = self::getUsage($userId);
        $limitStorage = ((int)$plan['max_storage_gb']) * 1024 * 1024 * 1024;
        if (($usage['storage_used_bytes'] + $newFileBytes) > $limitStorage) {
            return [
                'ok' => false,
                'code' => 'storage_limit',
                'message' => 'Limite de armazenamento do plano atingido.',
                'plan' => $plan,
                'usage' => $usage
            ];
        }

        return ['ok' => true, 'plan' => $plan, 'usage' => $usage];
    }

    public static function canStreamByOwner(int $ownerUserId): array
    {
        $plan = self::resolveActivePlan($ownerUserId);
        if (!$plan) {
            return [
                'ok' => false,
                'code' => 'no_plan',
                'message' => 'Conta sem plano ativo para reprodução.',
                'plan' => null
            ];
        }
        $usage = self::getUsage($ownerUserId);
        $limitBw = ((int)$plan['max_bandwidth_gb']) * 1024 * 1024 * 1024;
        if ($limitBw > 0 && $usage['bandwidth_used_bytes'] >= $limitBw) {
            return [
                'ok' => false,
                'code' => 'bandwidth_limit',
                'message' => 'Limite de banda mensal do plano foi atingido.',
                'plan' => $plan,
                'usage' => $usage
            ];
        }
        return ['ok' => true, 'plan' => $plan, 'usage' => $usage];
    }

    public static function registerBandwidth(int $userId, int $videoId, int $bytesSent, string $source = 'hls'): void
    {
        if ($bytesSent <= 0) return;
        $stmt = db()->prepare("
            INSERT INTO bandwidth_logs (user_id, video_id, bytes_sent, source, ip_address, user_agent)
            VALUES (:u, :v, :b, :s, :ip, :ua)
        ");
        $stmt->execute([
            'u' => $userId,
            'v' => $videoId,
            'b' => $bytesSent,
            's' => $source,
            'ip' => substr((string)($_SERVER['REMOTE_ADDR'] ?? ''), 0, 45),
            'ua' => substr((string)($_SERVER['HTTP_USER_AGENT'] ?? ''), 0, 255),
        ]);
    }

    public static function registerPlaybackStart(int $userId, int $videoId, int $estimatedBytes): void
    {
        db()->prepare("UPDATE videos SET playback_starts = playback_starts + 1 WHERE id=:id")->execute(['id' => $videoId]);
        // Fallback para players externos que não passam por endpoint HLS.
        if ($estimatedBytes > 0) {
            self::registerBandwidth($userId, $videoId, $estimatedBytes, 'play_start_estimate');
        }
    }

    public static function syncUserPlanFromSubscription(array $subRow): void
    {
        $userId = (int)($subRow['user_id'] ?? 0);
        $planId = (int)($subRow['plan_id'] ?? 0);
        $status = (string)($subRow['status'] ?? '');
        if (!$userId) return;

        if (in_array($status, ['active','authorized'], true) && $planId > 0) {
            db()->prepare("UPDATE users SET current_plan_id=:p, status='active' WHERE id=:u")
              ->execute(['p' => $planId, 'u' => $userId]);
            return;
        }

        // Se assinatura cancelada/expirada e não há outra ativa, remove plano atual
        if (in_array($status, ['canceled','expired','payment_failed'], true)) {
            $st = db()->prepare("SELECT id FROM subscriptions WHERE user_id=:u AND status IN ('active','authorized') ORDER BY id DESC LIMIT 1");
            $st->execute(['u' => $userId]);
            $alive = $st->fetch();
            if (!$alive) {
                db()->prepare("UPDATE users SET current_plan_id=NULL WHERE id=:u")->execute(['u' => $userId]);
            }
        }
    }
}
