<?php
declare(strict_types=1);
require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../helpers.php';
require_once __DIR__ . '/RetryProfileResolver.php';

class VideoPipelineService
{
    public static function enqueueTranscode(int $videoId): void
    {
        // evita fila duplicada pendente/processando
        $check = db()->prepare("
            SELECT id FROM video_jobs
            WHERE video_id=:v
              AND job_type='transcode_hls'
              AND status IN ('queued','processing')
            LIMIT 1
        ");
        $check->execute(['v' => $videoId]);
        if ($check->fetch()) return;

        $sql = "INSERT INTO video_jobs
                (video_id, job_type, status, priority, attempts, max_attempts, payload, run_after)
                VALUES (:v, 'transcode_hls', 'queued', 5, 0, :m, :p, NOW())";
        db()->prepare($sql)->execute([
            'v' => $videoId,
            'm' => VIDEO_WORKER_MAX_ATTEMPTS,
            'p' => json_encode(['trigger' => 'upload'], JSON_UNESCAPED_UNICODE),
        ]);

        db()->prepare("UPDATE videos SET status='processing' WHERE id=:id")->execute(['id' => $videoId]);
    }

    public static function processNextJob(): array
    {
        // Claim otimista + transação
        $pdo = db();
        $pdo->beginTransaction();
        try {
            $stmt = $pdo->prepare("
                SELECT * FROM video_jobs
                WHERE status='queued'
                  AND run_after <= NOW()
                ORDER BY priority ASC, id ASC
                LIMIT 1
                FOR UPDATE
            ");
            $stmt->execute();
            $job = $stmt->fetch();
            if (!$job) {
                $pdo->commit();
                return ['ok' => true, 'message' => 'queue_empty'];
            }

            $pdo->prepare("UPDATE video_jobs SET status='processing', started_at=NOW(), attempts=attempts+1 WHERE id=:id")
                ->execute(['id' => $job['id']]);
            $pdo->commit();

            $result = self::runJob($job);

            if ($result['ok']) {
                $pdo->prepare("UPDATE video_jobs SET status='done', finished_at=NOW(), error_message=NULL WHERE id=:id")
                    ->execute(['id' => $job['id']]);
            } else {
                $failMsg = self::tailText((string)($result['message'] ?? 'Erro desconhecido'), 5000);

                // requeue se ainda houver tentativas
                $st = $pdo->prepare("SELECT attempts, max_attempts FROM video_jobs WHERE id=:id LIMIT 1");
                $st->execute(['id' => $job['id']]);
                $fresh = $st->fetch();
                $attempts = (int)($fresh['attempts'] ?? 1);
                $maxAttempts = (int)($fresh['max_attempts'] ?? VIDEO_WORKER_MAX_ATTEMPTS);

                if ($attempts < $maxAttempts) {
                    $pdo->prepare("
                        UPDATE video_jobs
                        SET status='queued',
                            error_message=:e,
                            run_after=DATE_ADD(NOW(), INTERVAL 1 MINUTE)
                        WHERE id=:id
                    ")->execute(['e' => $failMsg, 'id' => $job['id']]);
                } else {
                    $pdo->prepare("UPDATE video_jobs SET status='failed', finished_at=NOW(), error_message=:e WHERE id=:id")
                        ->execute(['e' => $failMsg, 'id' => $job['id']]);
                    $pdo->prepare("UPDATE videos SET status='failed', processing_error=:e WHERE id=:v")
                        ->execute(['e' => $failMsg, 'v' => $job['video_id']]);
                }
            }

            return $result;
        } catch (Throwable $e) {
            if ($pdo->inTransaction()) $pdo->rollBack();
            return ['ok' => false, 'message' => 'Job crash: ' . $e->getMessage()];
        }
    }

    private static function runJob(array $job): array
    {
        $videoId = (int)$job['video_id'];
        $jobId = (int)$job['id'];

        $st = db()->prepare("SELECT * FROM videos WHERE id=:id LIMIT 1");
        $st->execute(['id' => $videoId]);
        $video = $st->fetch();
        if (!$video) return ['ok' => false, 'message' => 'Vídeo não encontrado'];

        if ($video['storage_type'] !== 'local') {
            db()->prepare("UPDATE videos SET status='ready' WHERE id=:id")->execute(['id' => $videoId]);
            return ['ok' => true, 'message' => 'Vídeo externo sem transcodificação'];
        }

        $inputAbs = dirname(__DIR__, 2) . '/public/' . ltrim((string)$video['video_path'], '/');
        if (!file_exists($inputAbs)) {
            return ['ok' => false, 'message' => 'Arquivo original não encontrado: ' . $inputAbs];
        }

        $ff = ffmpeg_health();
        if (!$ff['ok']) {
            return [
                'ok' => false,
                'message' => $ff['message']
                    . ' ffmpeg=' . ($ff['ffmpeg']['resolved'] ?? ($ff['ffmpeg']['bin'] ?? ''))
                    . ' | ffprobe=' . ($ff['ffprobe']['resolved'] ?? ($ff['ffprobe']['bin'] ?? ''))
            ];
        }

        $duration = self::probeDuration($inputAbs);
        if ($duration !== null) {
            db()->prepare("UPDATE videos SET source_duration_seconds=:d WHERE id=:id")->execute(['d' => $duration, 'id' => $videoId]);
        }

        // Limite de duração por plano
        $plan = self::resolvePlanForUser((int)$video['user_id']);
        if ($plan && $duration !== null) {
            $maxSec = ((int)$plan['max_video_duration_min']) * 60;
            if ($duration > $maxSec) {
                return ['ok' => false, 'message' => 'Duração acima do limite do plano'];
            }
        }

        // pipeline HLS
        $outDir = ensure_hls_video_dir($videoId);
        self::rrmdirFilesOnly($outDir);

        $profiles = self::profilesByInput($inputAbs);

        // Fallback resiliente:
        // - 360p é obrigatório para considerar sucesso
        // - Falhas em 720p/1080p não derrubam o job inteiro se 360p estiver pronto
        $renditions = [];
        $failedRenditions = [];
        $has360p = false;

        foreach ($profiles as $pf) {
            $rendRes = self::createHlsVariantWithRetry($inputAbs, $outDir, $pf, $jobId);
            if (!$rendRes['ok']) {
                $failedRenditions[] = [
                    'label' => (string)($pf['label'] ?? 'unknown'),
                    'message' => (string)($rendRes['message'] ?? 'Falha sem detalhe'),
                ];
                continue;
            }

            $renditions[] = $rendRes['data'];
            if ((string)($rendRes['data']['label'] ?? '') === '360p') {
                $has360p = true;
            }
        }

        if (empty($renditions)) {
            $errs = [];
            foreach ($failedRenditions as $fr) {
                $errs[] = $fr['label'] . ': ' . self::tailText($fr['message'], 800);
            }
            return ['ok' => false, 'message' => 'Falha no ffmpeg. Nenhuma qualidade foi gerada. ' . implode(' | ', $errs)];
        }

        if (!$has360p) {
            $errs = [];
            foreach ($failedRenditions as $fr) {
                $errs[] = $fr['label'] . ': ' . self::tailText($fr['message'], 800);
            }
            return ['ok' => false, 'message' => 'Falha obrigatória na 360p. ' . implode(' | ', $errs)];
        }

        $masterPath = self::createMasterPlaylist($outDir, $renditions);
        $relativeMaster = 'hls/' . $videoId . '/master.m3u8';

        // persist renditions
        db()->prepare("DELETE FROM video_renditions WHERE video_id=:v")->execute(['v' => $videoId]);
        $ins = db()->prepare("
            INSERT INTO video_renditions
            (video_id, label, width, height, bitrate_kbps, playlist_file, status)
            VALUES (:v,:l,:w,:h,:b,:f,'ready')
        ");
        foreach ($renditions as $r) {
            $ins->execute([
                'v' => $videoId,
                'l' => $r['label'],
                'w' => $r['width'],
                'h' => $r['height'],
                'b' => $r['bitrate_kbps'],
                'f' => $r['playlist_file']
            ]);
        }

        db()->prepare("
            UPDATE videos
            SET status='ready',
                hls_master_path=:m,
                processing_error=NULL
            WHERE id=:id
        ")->execute(['m' => $relativeMaster, 'id' => $videoId]);

        $msg = 'Transcodificação concluída';
        if (!empty($failedRenditions)) {
            $labels = array_map(static fn($e) => (string)$e['label'], $failedRenditions);
            $msg = 'Transcodificação concluída com fallback. Falharam: ' . implode(', ', $labels);
        }
        return ['ok' => true, 'message' => $msg, 'master' => $masterPath];
    }

    private static function probeDuration(string $input): ?int
    {
        $cmd = escapeshellcmd(FFPROBE_BIN)
             . ' -v error -show_entries format=duration -of default=noprint_wrappers=1:nokey=1 '
             . escapeshellarg($input)
             . ' 2>&1';
        $out = shell_exec($cmd);
        if (!is_string($out)) return null;
        $out = trim($out);
        if ($out === '' || !is_numeric($out)) return null;
        return (int)round((float)$out);
    }

    private static function profilesByInput(string $input): array
    {
        // Perfil padrão para ambiente compartilhado:
        // 360p e 720p (1080p opcional se origem >=1080)
        $w = self::probeWidth($input);
        $profiles = [
            ['label' => '360p', 'width' => 640, 'height' => 360, 'video_k' => 800, 'audio_k' => 96],
            ['label' => '720p', 'width' => 1280, 'height' => 720, 'video_k' => 2800, 'audio_k' => 128],
        ];
        if ($w !== null && $w >= 1900) {
            $profiles[] = ['label' => '1080p', 'width' => 1920, 'height' => 1080, 'video_k' => 5000, 'audio_k' => 160];
        }
        return $profiles;
    }

    private static function probeWidth(string $input): ?int
    {
        $cmd = escapeshellcmd(FFPROBE_BIN)
             . ' -v error -select_streams v:0 -show_entries stream=width -of csv=s=x:p=0 '
             . escapeshellarg($input) . ' 2>&1';
        $out = shell_exec($cmd);
        if (!is_string($out)) return null;
        $out = trim($out);
        if ($out === '' || !preg_match('/^\d+$/', $out)) return null;
        return (int)$out;
    }

    private static function createHlsVariantWithRetry(string $input, string $outDir, array $pf, int $jobId = 0): array
    {
        $label = (string)($pf['label'] ?? '360p');
        $maxAttempts = 3;
        $lastMsg = 'Falha sem detalhe';

        for ($attempt = 0; $attempt < $maxAttempts; $attempt++) {
            $profile = RetryProfileResolver::resolve($pf, $attempt);
            $res = self::createHlsVariantAttempt($input, $outDir, $profile, $attempt, $jobId);
            if ($res['ok']) {
                return $res;
            }
            $lastMsg = (string)($res['message'] ?? $lastMsg);
        }

        return ['ok' => false, 'message' => "Falha no ffmpeg ({$label}) após {$maxAttempts} tentativas. {$lastMsg}"];
    }

    private static function createHlsVariantAttempt(string $input, string $outDir, array $pf, int $attempt, int $jobId): array
    {
        $label = (string)($pf['label'] ?? '360p');
        $plName = $label . '.m3u8';
        $segmentPatternPath = $outDir . '/' . $label . '_%03d.ts';

        // cmd.exe no Windows interpreta %...%; duplicar
        if (DIRECTORY_SEPARATOR === '\\') {
            $segmentPatternPath = str_replace('%', '%%', $segmentPatternPath);
        }

        $ffmpeg = escapeshellcmd((string)FFMPEG_BIN);
        $in = escapeshellarg($input);
        $playlist = escapeshellarg($outDir . '/' . $plName);
        $segPattern = escapeshellarg($segmentPatternPath);

        $width = (int)($pf['width'] ?? 640);
        $height = (int)($pf['height'] ?? 360);
        $videoK = (int)($pf['video_k'] ?? 800);
        $audioK = (int)($pf['audio_k'] ?? 96);
        $preset = (string)($pf['preset'] ?? 'veryfast');
        $crf = (int)($pf['crf'] ?? 20);

        $maxrate = (int)max(300, round($videoK * 1.07));
        $bufsize = (int)max(600, round($videoK * 1.5));

        $vf = "scale=w={$width}:h={$height}:force_original_aspect_ratio=decrease,pad={$width}:{$height}:(ow-iw)/2:(oh-ih)/2,format=yuv420p";

        $cmd = $ffmpeg
            . " -y -hide_banner -loglevel error"
            . " -i {$in}"
            . " -map 0:v:0 -map 0:a:0?"
            . " -vf " . escapeshellarg($vf)
            . " -c:v libx264 -preset {$preset} -crf {$crf} -sc_threshold 0"
            . " -g 48 -keyint_min 48"
            . " -b:v {$videoK}k -maxrate {$maxrate}k -bufsize {$bufsize}k"
            . " -pix_fmt yuv420p"
            . " -c:a aac -ar 48000 -b:a {$audioK}k -ac 2"
            . " -hls_time " . (int)HLS_SEGMENT_SECONDS
            . " -hls_playlist_type vod"
            . " -hls_flags independent_segments"
            . " -hls_segment_filename {$segPattern}"
            . " {$playlist} 2>&1";

        $output = [];
        $exitCode = 0;
        exec($cmd, $output, $exitCode);
        $raw = implode(PHP_EOL, $output);
        $tail = self::tailText($raw, 3500);

        self::logRetryAttempt($jobId, $label, $attempt, $exitCode, $tail, $videoK, $audioK, $preset, $crf);

        $playlistAbs = $outDir . '/' . $plName;
        if ($exitCode !== 0 || !file_exists($playlistAbs) || filesize($playlistAbs) <= 0) {
            $msg = "exit={$exitCode}; " . ($tail !== '' ? $tail : 'sem stderr');
            return ['ok' => false, 'message' => $msg];
        }

        return [
            'ok' => true,
            'data' => [
                'label' => $label,
                'width' => $width,
                'height' => $height,
                'bitrate_kbps' => $videoK,
                'playlist_file' => $plName,
            ]
        ];
    }

    private static function logRetryAttempt(
        int $jobId,
        string $label,
        int $attempt,
        int $exitCode,
        string $errorTail,
        int $videoK,
        int $audioK,
        string $preset,
        int $crf
    ): void {
        if ($jobId <= 0) return;
        try {
            $sql = "INSERT INTO video_retry_attempt_logs
                    (video_job_id, quality, attempt_no, exit_code, error_tail, profile_json, created_at)
                    VALUES (:j,:q,:a,:x,:e,:p,NOW())";
            $profile = json_encode([
                'video_k' => $videoK,
                'audio_k' => $audioK,
                'preset' => $preset,
                'crf' => $crf,
            ], JSON_UNESCAPED_UNICODE);

            db()->prepare($sql)->execute([
                'j' => $jobId,
                'q' => $label,
                'a' => $attempt,
                'x' => $exitCode,
                'e' => self::tailText($errorTail, 3500),
                'p' => $profile,
            ]);
        } catch (Throwable $e) {
            // não interromper pipeline por falha de log
        }
    }

    private static function createMasterPlaylist(string $outDir, array $renditions): string
    {
        $lines = ['#EXTM3U', '#EXT-X-VERSION:3'];
        foreach ($renditions as $r) {
            $band = (int)$r['bitrate_kbps'] * 1000;
            $lines[] = "#EXT-X-STREAM-INF:BANDWIDTH={$band},RESOLUTION={$r['width']}x{$r['height']}";
            $lines[] = $r['playlist_file'];
        }
        $master = implode("\n", $lines) . "\n";
        $masterPath = $outDir . '/master.m3u8';
        file_put_contents($masterPath, $master);
        return $masterPath;
    }

    private static function rrmdirFilesOnly(string $dir): void
    {
        if (!is_dir($dir)) return;
        $files = scandir($dir);
        if (!$files) return;
        foreach ($files as $f) {
            if ($f === '.' || $f === '..') continue;
            $p = $dir . '/' . $f;
            if (is_file($p)) @unlink($p);
        }
    }

    private static function resolvePlanForUser(int $userId): ?array
    {
        $stmt = db()->prepare("
            SELECT p.*
            FROM users u
            JOIN plans p ON p.id = u.current_plan_id
            WHERE u.id=:u
            LIMIT 1
        ");
        $stmt->execute(['u' => $userId]);
        $p = $stmt->fetch();
        if ($p) return $p;

        $stmt = db()->prepare("
            SELECT p.*
            FROM subscriptions s
            JOIN plans p ON p.id=s.plan_id
            WHERE s.user_id=:u
              AND s.status IN ('active','authorized')
            ORDER BY s.id DESC LIMIT 1
        ");
        $stmt->execute(['u' => $userId]);
        return $stmt->fetch() ?: null;
    }

    private static function tailText(string $text, int $max = 1500): string
    {
        $text = trim($text);
        if ($text === '') return '';
        if (function_exists('mb_strlen') && function_exists('mb_substr')) {
            if (mb_strlen($text) <= $max) return $text;
            return mb_substr($text, -$max);
        }
        if (strlen($text) <= $max) return $text;
        return substr($text, -$max);
    }
}
