<?php
declare(strict_types=1);

$autoloadPath = dirname(__DIR__) . '/vendor/autoload.php';
if (file_exists($autoloadPath)) {
    require_once $autoloadPath;
}

function env(string $key, ?string $default = null): ?string {
    static $vars = null;
    if ($vars === null) {
        $vars = [];
        $envPath = dirname(__DIR__) . '/.env';
        if (file_exists($envPath)) {
            $lines = file($envPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) ?: [];
            foreach ($lines as $line) {
                $line = trim($line);
                if ($line === '' || str_starts_with($line, '#')) continue;
                $eqPos = strpos($line, '=');
                if ($eqPos === false) continue;
                $k = trim(substr($line, 0, $eqPos));
                $v = trim(substr($line, $eqPos + 1));
                $vars[$k] = $v;
            }
        }
    }
    return $_ENV[$key] ?? $_SERVER[$key] ?? ($vars[$key] ?? $default);
}


function detect_app_base_path(): string {
    $configured = trim((string)env('APP_BASE_PATH', ''));
    if ($configured !== '') {
        $configured = '/' . trim($configured, '/');
        return $configured === '/' ? '' : $configured;
    }

    $script = (string)($_SERVER['SCRIPT_NAME'] ?? '');
    $pos = strpos($script, '/public/');
    if ($pos !== false) {
        return rtrim(substr($script, 0, $pos + 7), '/');
    }
    if (preg_match('#^(.*?/public)$#', $script, $m)) {
        return rtrim($m[1], '/');
    }
    return '';
}

function detect_default_app_url(): string {
    if (PHP_SAPI === 'cli') {
        return 'http://localhost';
    }
    $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $base = detect_app_base_path();
    return $scheme . '://' . $host . $base;
}


date_default_timezone_set('America/Sao_Paulo');

define('APP_NAME', env('APP_NAME', 'VideoSaaS'));
define('APP_BASE_PATH', detect_app_base_path());
define('APP_URL', rtrim((string)env('APP_URL', detect_default_app_url()), '/'));
define('APP_ENV', env('APP_ENV', 'local'));
define('APP_DEBUG', filter_var(env('APP_DEBUG', 'true'), FILTER_VALIDATE_BOOL));
define('APP_KEY', env('APP_KEY', ''));
define('UPLOAD_PATH', env('UPLOAD_PATH', '../storage/uploads'));
define('MAX_UPLOAD_MB', (int)env('MAX_UPLOAD_MB', '2048'));

define('DEFAULT_CURRENCY', env('DEFAULT_CURRENCY', 'BRL'));

/** Video pipeline */
define('FFMPEG_BIN', env('FFMPEG_BIN', 'ffmpeg'));
define('FFPROBE_BIN', env('FFPROBE_BIN', 'ffprobe'));
define('HLS_SEGMENT_SECONDS', max(2, (int)env('HLS_SEGMENT_SECONDS', '6')));
define('VIDEO_WORKER_MAX_ATTEMPTS', max(1, (int)env('VIDEO_WORKER_MAX_ATTEMPTS', '3')));

define('MP_ACCESS_TOKEN', env('MP_ACCESS_TOKEN', ''));
define('MP_WEBHOOK_SECRET', env('MP_WEBHOOK_SECRET', ''));
define('MP_API_BASE', env('MP_API_BASE', 'https://api.mercadopago.com'));

define('PB_ACCESS_TOKEN', env('PB_ACCESS_TOKEN', ''));
define('PB_API_BASE', env('PB_API_BASE', 'https://sandbox.api.assinaturas.pagseguro.com'));
define('PB_WEBHOOK_VALIDATE_ENDPOINT', env('PB_WEBHOOK_VALIDATE_ENDPOINT', 'https://sandbox.api.pagseguro.com/payments/notifications/verify'));
define('PB_WEBHOOK_TOKEN', env('PB_WEBHOOK_TOKEN', ''));

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
