<?php
declare(strict_types=1);
require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../app/helpers.php';

require_admin();

function calc_period_end_manual(string $intervalUnit, int $intervalCount): string {
    $intervalCount = max(1, $intervalCount);
    $start = new DateTimeImmutable('now');
    switch ($intervalUnit) {
        case 'day': $end = $start->modify('+' . $intervalCount . ' day'); break;
        case 'week': $end = $start->modify('+' . $intervalCount . ' week'); break;
        case 'year': $end = $start->modify('+' . $intervalCount . ' year'); break;
        case 'month':
        default: $end = $start->modify('+' . $intervalCount . ' month'); break;
    }
    return $end->format('Y-m-d H:i:s');
}

if (is_post()) {
    if (!csrf_check($_POST['_csrf'] ?? null)) {
        flash('error', 'CSRF inválido.');
        redirect('/admin/manual_payments.php');
    }

    $action = (string)($_POST['action'] ?? '');
    $proofId = (int)($_POST['proof_id'] ?? 0);
    $note = trim((string)($_POST['admin_note'] ?? ''));

    if ($proofId <= 0) {
        flash('error', 'Comprovante inválido.');
        redirect('/admin/manual_payments.php');
    }

    $q = db()->prepare("
        SELECT mpp.*, s.id AS sub_id, s.user_id, s.plan_id, s.provider, s.status AS sub_status,
               p.interval_unit, p.interval_count
        FROM manual_payment_proofs mpp
        JOIN subscriptions s ON s.id = mpp.subscription_id
        JOIN plans p ON p.id = s.plan_id
        WHERE mpp.id=:id
        LIMIT 1
    ");
    $q->execute(['id' => $proofId]);
    $row = $q->fetch();

    if (!$row) {
        flash('error', 'Comprovante não encontrado.');
        redirect('/admin/manual_payments.php');
    }

    if (($row['provider'] ?? '') !== 'manual') {
        flash('error', 'Este comprovante não está vinculado a pagamento manual.');
        redirect('/admin/manual_payments.php');
    }

    if ($action === 'approve') {
        $end = calc_period_end_manual((string)$row['interval_unit'], (int)$row['interval_count']);

        db()->beginTransaction();
        try {
            db()->prepare("UPDATE manual_payment_proofs
                           SET status='approved', admin_note=:note, reviewed_at=NOW(), reviewed_by=:rid, updated_at=NOW()
                           WHERE id=:id")
              ->execute([
                  'note' => $note !== '' ? $note : null,
                  'rid' => (int)(auth_user()['id'] ?? 0),
                  'id' => $proofId
              ]);

            db()->prepare("UPDATE subscriptions
                           SET status='authorized',
                               started_at=COALESCE(started_at, NOW()),
                               current_period_start=COALESCE(current_period_start, NOW()),
                               current_period_end=:end,
                               canceled_at=NULL,
                               updated_at=NOW()
                           WHERE id=:sid")
              ->execute([
                  'end' => $end,
                  'sid' => (int)$row['subscription_id']
              ]);

            db()->prepare("UPDATE users SET current_plan_id=:pid, status='active' WHERE id=:uid")
              ->execute([
                  'pid' => (int)$row['plan_id'],
                  'uid' => (int)$row['user_id']
              ]);

            db()->commit();
            flash('success', 'Comprovante aprovado e assinatura liberada.');
        } catch (Throwable $e) {
            if (db()->inTransaction()) db()->rollBack();
            flash('error', 'Falha ao aprovar comprovante: ' . $e->getMessage());
        }
        redirect('/admin/manual_payments.php');
    }

    if ($action === 'reject') {
        db()->beginTransaction();
        try {
            db()->prepare("UPDATE manual_payment_proofs
                           SET status='rejected', admin_note=:note, reviewed_at=NOW(), reviewed_by=:rid, updated_at=NOW()
                           WHERE id=:id")
              ->execute([
                  'note' => $note !== '' ? $note : null,
                  'rid' => (int)(auth_user()['id'] ?? 0),
                  'id' => $proofId
              ]);

            db()->prepare("UPDATE subscriptions
                           SET status='canceled', canceled_at=NOW(), updated_at=NOW()
                           WHERE id=:sid")
              ->execute(['sid' => (int)$row['subscription_id']]);

            db()->commit();
            flash('success', 'Comprovante reprovado e assinatura cancelada.');
        } catch (Throwable $e) {
            if (db()->inTransaction()) db()->rollBack();
            flash('error', 'Falha ao reprovar comprovante: ' . $e->getMessage());
        }
        redirect('/admin/manual_payments.php');
    }

    flash('error', 'Ação inválida.');
    redirect('/admin/manual_payments.php');
}

$status = (string)($_GET['status'] ?? '');
$q = trim((string)($_GET['q'] ?? ''));

$where = ['1=1'];
$params = [];

$allowed = ['pending','approved','rejected'];
if (in_array($status, $allowed, true)) {
    $where[] = "mpp.status=:status";
    $params['status'] = $status;
} else {
    $status = '';
}

if ($q !== '') {
    $where[] = "(u.name LIKE :q OR u.email LIKE :q OR p.name LIKE :q OR s.external_reference LIKE :q OR mpp.original_name LIKE :q)";
    $params['q'] = '%' . $q . '%';
}

$sql = "
SELECT mpp.*,
       s.id AS subscription_id,
       s.status AS subscription_status,
       s.external_reference,
       u.name AS user_name,
       u.email AS user_email,
       p.name AS plan_name
FROM manual_payment_proofs mpp
JOIN subscriptions s ON s.id = mpp.subscription_id
JOIN users u ON u.id = s.user_id
JOIN plans p ON p.id = s.plan_id
WHERE " . implode(' AND ', $where) . "
ORDER BY mpp.id DESC
LIMIT 500
";
$st = db()->prepare($sql);
$st->execute($params);
$rows = $st->fetchAll();

$stats = db()->query("
SELECT
  COUNT(*) total,
  SUM(CASE WHEN status='pending' THEN 1 ELSE 0 END) pending_count,
  SUM(CASE WHEN status='approved' THEN 1 ELSE 0 END) approved_count,
  SUM(CASE WHEN status='rejected' THEN 1 ELSE 0 END) rejected_count
FROM manual_payment_proofs
")->fetch();

$pageTitle = 'Admin - Comprovantes manuais';
include __DIR__ . '/../partials/head.php';
include __DIR__ . '/../partials/navbar.php';
include __DIR__ . '/../partials/sidebar.php';
?>
<div class="content-wrapper">
  <div class="content-header">
    <div class="container-fluid">
      <h1>Comprovantes de pagamento manual</h1>
      <p class="text-secondary mb-0">Aprove ou reprove comprovantes enviados pelos clientes.</p>
    </div>
  </div>

  <section class="content">
    <div class="container-fluid">
      <?php include __DIR__ . '/../partials/flash.php'; ?>

      <div class="row">
        <div class="col-md-3">
          <div class="small-box glass-card">
            <div class="inner"><h3><?= (int)($stats['total'] ?? 0) ?></h3><p>Total</p></div>
            <div class="icon"><i class="fas fa-file-invoice"></i></div>
          </div>
        </div>
        <div class="col-md-3">
          <div class="small-box glass-card">
            <div class="inner"><h3><?= (int)($stats['pending_count'] ?? 0) ?></h3><p>Pendentes</p></div>
            <div class="icon"><i class="fas fa-hourglass-half"></i></div>
          </div>
        </div>
        <div class="col-md-3">
          <div class="small-box glass-card">
            <div class="inner"><h3><?= (int)($stats['approved_count'] ?? 0) ?></h3><p>Aprovados</p></div>
            <div class="icon"><i class="fas fa-circle-check"></i></div>
          </div>
        </div>
        <div class="col-md-3">
          <div class="small-box glass-card">
            <div class="inner"><h3><?= (int)($stats['rejected_count'] ?? 0) ?></h3><p>Reprovados</p></div>
            <div class="icon"><i class="fas fa-circle-xmark"></i></div>
          </div>
        </div>
      </div>

      <div class="card glass-card mb-3">
        <div class="card-body">
          <form method="get" class="row g-2 align-items-end">
            <div class="col-md-3">
              <label class="form-label">Status</label>
              <select name="status" class="form-select bg-transparent text-light">
                <option value="">Todos</option>
                <?php foreach ($allowed as $s): ?>
                  <option value="<?= e($s) ?>" <?= $status === $s ? 'selected' : '' ?>><?= e($s) ?></option>
                <?php endforeach; ?>
              </select>
            </div>
            <div class="col-md-7">
              <label class="form-label">Busca</label>
              <input type="text" name="q" value="<?= e($q) ?>" class="form-control bg-transparent text-light" placeholder="nome, email, plano, referência, arquivo...">
            </div>
            <div class="col-md-2 d-grid">
              <button class="btn btn-neon">Filtrar</button>
            </div>
          </form>
        </div>
      </div>

      <div class="card glass-card">
        <div class="card-body table-responsive">
          <table class="table table-dark table-hover align-middle">
            <thead>
              <tr>
                <th>#</th>
                <th>Cliente</th>
                <th>Plano/Assinatura</th>
                <th>Arquivo</th>
                <th>Status</th>
                <th>Enviado em</th>
                <th>Revisão</th>
                <th>Ações</th>
              </tr>
            </thead>
            <tbody>
              <?php if (!$rows): ?>
                <tr><td colspan="8" class="text-center text-secondary">Nenhum comprovante encontrado.</td></tr>
              <?php endif; ?>

              <?php foreach ($rows as $r): ?>
                <?php
                  $stClass = 'secondary';
                  if ((string)$r['status'] === 'pending') $stClass = 'warning text-dark';
                  if ((string)$r['status'] === 'approved') $stClass = 'success';
                  if ((string)$r['status'] === 'rejected') $stClass = 'danger';
                ?>
                <tr>
                  <td><?= (int)$r['id'] ?></td>
                  <td>
                    <div><?= e((string)$r['user_name']) ?></div>
                    <small class="text-secondary"><?= e((string)$r['user_email']) ?></small>
                  </td>
                  <td>
                    <div><?= e((string)$r['plan_name']) ?></div>
                    <small class="text-secondary">Sub #<?= (int)$r['subscription_id'] ?> · <?= e((string)$r['subscription_status']) ?></small>
                    <?php if (!empty($r['external_reference'])): ?>
                      <div><small class="text-secondary">Ref: <code><?= e((string)$r['external_reference']) ?></code></small></div>
                    <?php endif; ?>
                  </td>
                  <td>
                    <div><?= e((string)($r['original_name'] ?: basename((string)$r['file_path']))) ?></div>
                    <small class="text-secondary"><?= e((string)$r['mime_type']) ?> · <?= number_format(((int)$r['file_size']) / 1024, 1, ',', '.') ?> KB</small>
                    <div class="mt-1">
                      <a href="<?= e(url('/admin/manual_proof_file.php?id=' . (int)$r['id'])) ?>" target="_blank" class="btn btn-outline-info btn-sm">
                        <i class="fas fa-eye"></i> Ver arquivo
                      </a>
                    </div>
                  </td>
                  <td><span class="badge bg-<?= e($stClass) ?>"><?= e((string)$r['status']) ?></span></td>
                  <td><?= e((string)$r['submitted_at']) ?></td>
                  <td>
                    <?php if (!empty($r['reviewed_at'])): ?>
                      <div><?= e((string)$r['reviewed_at']) ?></div>
                    <?php else: ?>
                      <span class="text-secondary">-</span>
                    <?php endif; ?>
                    <?php if (!empty($r['admin_note'])): ?>
                      <small class="d-block text-warning"><?= e((string)$r['admin_note']) ?></small>
                    <?php endif; ?>
                  </td>
                  <td>
                    <?php if ((string)$r['status'] === 'pending'): ?>
                      <form method="post" class="mb-1">
                        <input type="hidden" name="_csrf" value="<?= e(csrf_token()) ?>">
                        <input type="hidden" name="proof_id" value="<?= (int)$r['id'] ?>">
                        <input type="hidden" name="action" value="approve">
                        <input type="text" name="admin_note" class="form-control form-control-sm mb-1 bg-transparent text-light" placeholder="Observação opcional">
                        <button class="btn btn-success btn-sm w-100" onclick="return confirm('Aprovar comprovante e liberar plano?')">
                          <i class="fas fa-check"></i> Aprovar
                        </button>
                      </form>
                      <form method="post">
                        <input type="hidden" name="_csrf" value="<?= e(csrf_token()) ?>">
                        <input type="hidden" name="proof_id" value="<?= (int)$r['id'] ?>">
                        <input type="hidden" name="action" value="reject">
                        <input type="text" name="admin_note" class="form-control form-control-sm mb-1 bg-transparent text-light" placeholder="Motivo da reprovação">
                        <button class="btn btn-outline-danger btn-sm w-100" onclick="return confirm('Reprovar comprovante e cancelar assinatura?')">
                          <i class="fas fa-xmark"></i> Reprovar
                        </button>
                      </form>
                    <?php else: ?>
                      <span class="text-secondary">Revisado</span>
                    <?php endif; ?>
                  </td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
      </div>

      <small class="text-secondary d-block mt-2">Exibindo no máximo 500 registros por página.</small>
    </div>
  </section>
</div>
<?php include __DIR__ . '/../partials/footer.php'; ?>
