<?php
declare(strict_types=1);
require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../app/helpers.php';
require_admin();

function decode_extra(?string $json): array {
    if (!$json) return [];
    $tmp = json_decode($json, true);
    return is_array($tmp) ? $tmp : [];
}

if (is_post()) {
    if (!csrf_check($_POST['_csrf'] ?? null)) {
        flash('error','CSRF inválido');
        redirect('/admin/payments.php');
    }

    $provider = (string)($_POST['provider'] ?? '');
    $allowed = ['mercadopago','pagbank','manual'];
    if (!in_array($provider, $allowed, true)) {
        flash('error','Provedor inválido');
        redirect('/admin/payments.php');
    }

    $isEnabled = isset($_POST['is_enabled']) ? 1 : 0;
    $env = ($_POST['environment'] ?? 'sandbox') === 'production' ? 'production' : 'sandbox';
    $accessToken = trim((string)($_POST['access_token'] ?? ''));
    $publicKey = trim((string)($_POST['public_key'] ?? ''));
    $webhookSecret = trim((string)($_POST['webhook_secret'] ?? ''));
    $extra = trim((string)($_POST['extra_json'] ?? ''));

    if ($provider === 'manual') {
        $env = ($_POST['environment'] ?? 'production') === 'sandbox' ? 'sandbox' : 'production';
        $accessToken = '';
        $publicKey = '';
        $webhookSecret = '';
        $extraData = [
            'instructions' => trim((string)($_POST['manual_instructions'] ?? '')),
            'pix_key' => trim((string)($_POST['manual_pix_key'] ?? '')),
            'whatsapp' => trim((string)($_POST['manual_whatsapp'] ?? '')),
            'bank_details' => trim((string)($_POST['manual_bank_details'] ?? '')),
            'auto_approve' => isset($_POST['manual_auto_approve']) ? 1 : 0,
        ];
        $extra = json_encode($extraData, JSON_UNESCAPED_UNICODE);
    }

    $sql = "UPDATE payment_settings
            SET is_enabled=:en, environment=:env, access_token=:token, public_key=:pk, webhook_secret=:ws, extra_json=:extra
            WHERE provider=:p";
    $st = db()->prepare($sql);
    $st->execute([
      'en' => $isEnabled,
      'env' => $env,
      'token' => $accessToken ?: null,
      'pk' => $publicKey ?: null,
      'ws' => $webhookSecret ?: null,
      'extra' => $extra !== '' ? $extra : null,
      'p' => $provider
    ]);

    if ($st->rowCount() === 0) {
        flash('error', 'Configuração não encontrada para este provedor. Rode a migration 003 para habilitar pagamento manual.');
        redirect('/admin/payments.php');
    }

    flash('success', 'Configurações salvas.');
    redirect('/admin/payments.php');
}

$rows = db()->query("SELECT * FROM payment_settings ORDER BY provider")->fetchAll();
$settings = [];
foreach ($rows as $r) $settings[$r['provider']] = $r;

$manualExists = isset($settings['manual']);

$pageTitle = 'Admin - Pagamentos';
include __DIR__ . '/../partials/head.php';
include __DIR__ . '/../partials/navbar.php';
include __DIR__ . '/../partials/sidebar.php';
?>
<div class="content-wrapper">
  <div class="content-header"><div class="container-fluid"><h1>Pagamentos</h1></div></div>
  <section class="content"><div class="container-fluid">
    <?php include __DIR__ . '/../partials/flash.php'; ?>
    <div class="mb-2">
      <a href="<?= e(url('/admin/payments_list.php')) ?>" class="btn btn-outline-info btn-sm">
        <i class="fas fa-receipt"></i> Ver todos os pagamentos / aprovar manual
      </a>
    </div>


    <?php if (!$manualExists): ?>
      <div class="alert alert-warning">
        A opção <strong>Pagamento manual</strong> não apareceu porque sua base ainda não foi migrada.
        Rode: <code>sql/migrations/003_manual_payment_and_plan_assign.sql</code>
      </div>
    <?php endif; ?>

    <div class="row">
      <?php
      $providers = ['mercadopago'=>'Mercado Pago','pagbank'=>'PagBank'];
      if ($manualExists) $providers['manual'] = 'Pagamento Manual';
      foreach ($providers as $key => $label):
          $s = $settings[$key] ?? ['is_enabled'=>0,'environment'=>'sandbox','access_token'=>'','public_key'=>'','webhook_secret'=>'','extra_json'=>''];
          $isManual = ($key === 'manual');
          $mx = $isManual ? decode_extra((string)$s['extra_json']) : [];
      ?>
      <div class="<?= $isManual ? 'col-12' : 'col-lg-6' ?>">
        <div class="card glass-card">
          <div class="card-header d-flex justify-content-between align-items-center">
            <span><?= e($label) ?></span>
            <?php if ($isManual): ?><span class="badge bg-info">Sem gateway</span><?php endif; ?>
          </div>
          <div class="card-body">
            <form method="post">
              <input type="hidden" name="_csrf" value="<?= e(csrf_token()) ?>">
              <input type="hidden" name="provider" value="<?= e($key) ?>">

              <div class="form-check mb-3">
                <input class="form-check-input" type="checkbox" name="is_enabled" id="enabled_<?= e($key) ?>" <?= (int)$s['is_enabled']===1?'checked':'' ?>>
                <label class="form-check-label" for="enabled_<?= e($key) ?>">Habilitado</label>
              </div>

              <?php if (!$isManual): ?>
                <div class="mb-2">
                  <label>Ambiente</label>
                  <select class="form-select bg-transparent text-light" name="environment">
                    <option value="sandbox" <?= $s['environment']==='sandbox'?'selected':'' ?>>sandbox</option>
                    <option value="production" <?= $s['environment']==='production'?'selected':'' ?>>production</option>
                  </select>
                </div>
                <div class="mb-2">
                  <label>Access Token</label>
                  <input class="form-control bg-transparent text-light" name="access_token" value="<?= e((string)$s['access_token']) ?>">
                </div>
                <div class="mb-2">
                  <label>Public Key</label>
                  <input class="form-control bg-transparent text-light" name="public_key" value="<?= e((string)$s['public_key']) ?>">
                </div>
                <div class="mb-2">
                  <label>Webhook Secret</label>
                  <input class="form-control bg-transparent text-light" name="webhook_secret" value="<?= e((string)$s['webhook_secret']) ?>">
                </div>
                <div class="mb-2">
                  <label>Extra JSON</label>
                  <textarea class="form-control bg-transparent text-light" name="extra_json" rows="3"><?= e((string)$s['extra_json']) ?></textarea>
                </div>
              <?php else: ?>
                <div class="alert alert-secondary py-2">
                  Configure aqui as instruções de pagamento offline/manual (PIX, transferência, WhatsApp, etc.).
                </div>

                <input type="hidden" name="environment" value="<?= e((string)($s['environment'] ?: 'production')) ?>">

                <div class="mb-2">
                  <label>Instruções para o cliente</label>
                  <textarea class="form-control bg-transparent text-light" name="manual_instructions" rows="3" placeholder="Ex.: Faça o PIX e envie o comprovante no WhatsApp para liberar o plano."><?= e((string)($mx['instructions'] ?? '')) ?></textarea>
                </div>

                <div class="mb-2">
                  <label>Chave PIX</label>
                  <input class="form-control bg-transparent text-light" name="manual_pix_key" value="<?= e((string)($mx['pix_key'] ?? '')) ?>" placeholder="email@dominio.com ou chave aleatória">
                </div>

                <div class="mb-2">
                  <label>WhatsApp para envio do comprovante</label>
                  <input class="form-control bg-transparent text-light" name="manual_whatsapp" value="<?= e((string)($mx['whatsapp'] ?? '')) ?>" placeholder="5511999999999">
                </div>

                <div class="mb-2">
                  <label>Dados bancários (opcional)</label>
                  <textarea class="form-control bg-transparent text-light" name="manual_bank_details" rows="2" placeholder="Banco, agência, conta..."><?= e((string)($mx['bank_details'] ?? '')) ?></textarea>
                </div>

                <div class="form-check mb-2">
                  <input class="form-check-input" id="manual_auto_approve" type="checkbox" name="manual_auto_approve" <?= ((int)($mx['auto_approve'] ?? 0)===1?'checked':'') ?>>
                  <label class="form-check-label" for="manual_auto_approve">Auto-aprovar assinatura manual após clique do cliente</label>
                </div>
                <small class="text-secondary">Se desmarcado, o cliente ficará pendente e você ativa o plano manualmente em <strong>Gerenciar clientes</strong>.</small>
              <?php endif; ?>

              <button class="btn btn-neon mt-3">Salvar</button>
            </form>
          </div>
        </div>
      </div>
      <?php endforeach; ?>
    </div>

    <div class="card glass-card mt-2">
      <div class="card-body">
        <strong>Endpoints de webhook:</strong>
        <ul>
          <li>Mercado Pago: <code><?= e(app_url('/webhooks/mercadopago.php')) ?></code></li>
          <li>PagBank: <code><?= e(app_url('/webhooks/pagbank.php')) ?></code></li>
        </ul>
      </div>
    </div>
  </div></section>
</div>
<?php include __DIR__ . '/../partials/footer.php'; ?>
