<?php
declare(strict_types=1);
require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../app/helpers.php';
require_admin();

function calc_period_end(string $intervalUnit, int $intervalCount): string {
    $intervalCount = max(1, $intervalCount);
    $start = new DateTimeImmutable('now');
    switch ($intervalUnit) {
        case 'day': $end = $start->modify('+' . $intervalCount . ' day'); break;
        case 'week': $end = $start->modify('+' . $intervalCount . ' week'); break;
        case 'year': $end = $start->modify('+' . $intervalCount . ' year'); break;
        case 'month':
        default: $end = $start->modify('+' . $intervalCount . ' month'); break;
    }
    return $end->format('Y-m-d H:i:s');
}

if (is_post()) {
    if (!csrf_check($_POST['_csrf'] ?? null)) {
        flash('error', 'CSRF inválido.');
        redirect('/admin/payments_list.php');
    }

    $action = (string)($_POST['action'] ?? '');
    $subId = (int)($_POST['sub_id'] ?? 0);

    if ($subId <= 0) {
        flash('error', 'Assinatura inválida.');
        redirect('/admin/payments_list.php');
    }

    $st = db()->prepare("
        SELECT s.*, p.interval_unit, p.interval_count
        FROM subscriptions s
        JOIN plans p ON p.id = s.plan_id
        WHERE s.id=:id
        LIMIT 1
    ");
    $st->execute(['id' => $subId]);
    $sub = $st->fetch();

    if (!$sub) {
        flash('error', 'Assinatura não encontrada.');
        redirect('/admin/payments_list.php');
    }

    if ($action === 'approve_manual') {
        if (($sub['provider'] ?? '') !== 'manual') {
            flash('error', 'Somente pagamentos manuais podem ser aprovados aqui.');
            redirect('/admin/payments_list.php');
        }

        $end = calc_period_end((string)$sub['interval_unit'], (int)$sub['interval_count']);

        db()->prepare("
            UPDATE subscriptions
            SET status='authorized',
                started_at=COALESCE(started_at, NOW()),
                current_period_start=COALESCE(current_period_start, NOW()),
                current_period_end=:end,
                canceled_at=NULL,
                updated_at=NOW()
            WHERE id=:id
        ")->execute([
            'end' => $end,
            'id' => $subId
        ]);

        db()->prepare("UPDATE users SET current_plan_id=:pid, status='active' WHERE id=:uid")
          ->execute([
              'pid' => (int)$sub['plan_id'],
              'uid' => (int)$sub['user_id']
          ]);

        flash('success', 'Pagamento manual aprovado e plano liberado para o cliente.');
        redirect('/admin/payments_list.php');
    }

    if ($action === 'reject_manual') {
        if (($sub['provider'] ?? '') !== 'manual') {
            flash('error', 'Somente pagamentos manuais podem ser reprovados aqui.');
            redirect('/admin/payments_list.php');
        }

        db()->prepare("
            UPDATE subscriptions
            SET status='canceled',
                canceled_at=NOW(),
                updated_at=NOW()
            WHERE id=:id
        ")->execute(['id' => $subId]);

        flash('success', 'Pagamento manual reprovado/cancelado com sucesso.');
        redirect('/admin/payments_list.php');
    }

    flash('error', 'Ação inválida.');
    redirect('/admin/payments_list.php');
}

$provider = (string)($_GET['provider'] ?? '');
$status = (string)($_GET['status'] ?? '');
$q = trim((string)($_GET['q'] ?? ''));

$allowedProviders = ['mercadopago','pagbank','manual'];
$allowedStatus = ['pending','authorized','active','paused','canceled','expired','payment_failed'];

$where = ['1=1'];
$params = [];

if (in_array($provider, $allowedProviders, true)) {
    $where[] = "s.provider = :provider";
    $params['provider'] = $provider;
} else {
    $provider = '';
}

if (in_array($status, $allowedStatus, true)) {
    $where[] = "s.status = :status";
    $params['status'] = $status;
} else {
    $status = '';
}

if ($q !== '') {
    $where[] = "(u.name LIKE :q OR u.email LIKE :q OR s.external_reference LIKE :q OR s.provider_subscription_id LIKE :q)";
    $params['q'] = '%' . $q . '%';
}

$sql = "
    SELECT
        s.*,
        u.name AS user_name,
        u.email AS user_email,
        p.name AS plan_name,
        p.price AS plan_price,
        p.currency AS plan_currency,
        p.interval_unit,
        p.interval_count
    FROM subscriptions s
    JOIN users u ON u.id = s.user_id
    JOIN plans p ON p.id = s.plan_id
    WHERE " . implode(' AND ', $where) . "
    ORDER BY s.id DESC
    LIMIT 500
";
$st = db()->prepare($sql);
$st->execute($params);
$rows = $st->fetchAll();

$stats = db()->query("
    SELECT
      COUNT(*) AS total,
      SUM(CASE WHEN provider='manual' AND status='pending' THEN 1 ELSE 0 END) AS pending_manual,
      SUM(CASE WHEN status IN ('authorized','active') THEN 1 ELSE 0 END) AS paid_like,
      SUM(CASE WHEN status IN ('payment_failed','canceled','expired') THEN 1 ELSE 0 END) AS problem
    FROM subscriptions
")->fetch();

$pageTitle = 'Admin - Todos os Pagamentos';
include __DIR__ . '/../partials/head.php';
include __DIR__ . '/../partials/navbar.php';
include __DIR__ . '/../partials/sidebar.php';
?>
<div class="content-wrapper">
  <div class="content-header">
    <div class="container-fluid">
      <h1>Todos os pagamentos</h1>
      <p class="text-secondary mb-0">Aprove pagamentos manuais pendentes e acompanhe todas as assinaturas.</p>
    </div>
  </div>

  <section class="content">
    <div class="container-fluid">
      <?php include __DIR__ . '/../partials/flash.php'; ?>

      <div class="row">
        <div class="col-md-3">
          <div class="small-box glass-card">
            <div class="inner">
              <h3><?= (int)($stats['total'] ?? 0) ?></h3>
              <p>Total de pagamentos</p>
            </div>
            <div class="icon"><i class="fas fa-receipt"></i></div>
          </div>
        </div>
        <div class="col-md-3">
          <div class="small-box glass-card">
            <div class="inner">
              <h3><?= (int)($stats['pending_manual'] ?? 0) ?></h3>
              <p>Manual pendente</p>
            </div>
            <div class="icon"><i class="fas fa-hourglass-half"></i></div>
          </div>
        </div>
        <div class="col-md-3">
          <div class="small-box glass-card">
            <div class="inner">
              <h3><?= (int)($stats['paid_like'] ?? 0) ?></h3>
              <p>Autorizado/Ativo</p>
            </div>
            <div class="icon"><i class="fas fa-circle-check"></i></div>
          </div>
        </div>
        <div class="col-md-3">
          <div class="small-box glass-card">
            <div class="inner">
              <h3><?= (int)($stats['problem'] ?? 0) ?></h3>
              <p>Falhou/Cancelado</p>
            </div>
            <div class="icon"><i class="fas fa-triangle-exclamation"></i></div>
          </div>
        </div>
      </div>

      <div class="card glass-card mb-3">
        <div class="card-body">
          <form method="get" class="row g-2 align-items-end">
            <div class="col-md-3">
              <label class="form-label">Provedor</label>
              <select name="provider" class="form-select bg-transparent text-light">
                <option value="">Todos</option>
                <?php foreach ($allowedProviders as $p): ?>
                  <option value="<?= e($p) ?>" <?= $provider === $p ? 'selected' : '' ?>><?= e($p) ?></option>
                <?php endforeach; ?>
              </select>
            </div>
            <div class="col-md-3">
              <label class="form-label">Status</label>
              <select name="status" class="form-select bg-transparent text-light">
                <option value="">Todos</option>
                <?php foreach ($allowedStatus as $s): ?>
                  <option value="<?= e($s) ?>" <?= $status === $s ? 'selected' : '' ?>><?= e($s) ?></option>
                <?php endforeach; ?>
              </select>
            </div>
            <div class="col-md-4">
              <label class="form-label">Busca</label>
              <input type="text" name="q" value="<?= e($q) ?>" class="form-control bg-transparent text-light" placeholder="Nome, email, referência, ID externo...">
            </div>
            <div class="col-md-2 d-grid">
              <button class="btn btn-neon">Filtrar</button>
            </div>
          </form>

          <div class="mt-3">
            <a class="btn btn-outline-info btn-sm" href="<?= e(url('/admin/payments.php')) ?>">
              <i class="fas fa-gear"></i> Configurações de gateway
            </a>
            <a class="btn btn-outline-success btn-sm" href="<?= e(url('/admin/manual_payments.php')) ?>">
              <i class="fas fa-file-invoice-dollar"></i> Comprovantes manuais
            </a>
          </div>
        </div>
      </div>

      <div class="card glass-card">
        <div class="card-body table-responsive">
          <table class="table table-dark table-hover align-middle">
            <thead>
              <tr>
                <th>#</th>
                <th>Cliente</th>
                <th>Plano</th>
                <th>Valor</th>
                <th>Provedor</th>
                <th>Status</th>
                <th>Referência</th>
                <th>Criado em</th>
                <th>Ações</th>
              </tr>
            </thead>
            <tbody>
              <?php if (!$rows): ?>
                <tr><td colspan="9" class="text-center text-secondary">Nenhum pagamento encontrado.</td></tr>
              <?php endif; ?>

              <?php foreach ($rows as $r): ?>
                <?php
                  $isPendingManual = ((string)$r['provider'] === 'manual' && (string)$r['status'] === 'pending');
                  $statusClass = 'secondary';
                  if (in_array((string)$r['status'], ['authorized','active'], true)) $statusClass = 'success';
                  elseif (in_array((string)$r['status'], ['payment_failed','canceled','expired'], true)) $statusClass = 'danger';
                  elseif ((string)$r['status'] === 'pending') $statusClass = 'warning text-dark';
                ?>
                <tr>
                  <td><?= (int)$r['id'] ?></td>
                  <td>
                    <div><?= e((string)$r['user_name']) ?></div>
                    <small class="text-secondary"><?= e((string)$r['user_email']) ?></small>
                  </td>
                  <td><?= e((string)$r['plan_name']) ?></td>
                  <td><?= e((string)$r['plan_currency']) ?> <?= number_format((float)$r['plan_price'], 2, ',', '.') ?></td>
                  <td><span class="badge bg-info"><?= e((string)$r['provider']) ?></span></td>
                  <td><span class="badge bg-<?= $statusClass ?>"><?= e((string)$r['status']) ?></span></td>
                  <td>
                    <?php if (!empty($r['external_reference'])): ?>
                      <code><?= e((string)$r['external_reference']) ?></code>
                    <?php else: ?>
                      <span class="text-secondary">-</span>
                    <?php endif; ?>
                    <?php if (!empty($r['provider_subscription_id'])): ?>
                      <div><small class="text-secondary">SubID: <?= e((string)$r['provider_subscription_id']) ?></small></div>
                    <?php endif; ?>
                  </td>
                  <td><?= e((string)$r['created_at']) ?></td>
                  <td>
                    <?php if ($isPendingManual): ?>
                      <form method="post" class="d-inline">
                        <input type="hidden" name="_csrf" value="<?= e(csrf_token()) ?>">
                        <input type="hidden" name="action" value="approve_manual">
                        <input type="hidden" name="sub_id" value="<?= (int)$r['id'] ?>">
                        <button class="btn btn-success btn-sm" onclick="return confirm('Aprovar pagamento manual e liberar plano?')">
                          <i class="fas fa-check"></i> Aprovar
                        </button>
                      </form>
                      <form method="post" class="d-inline">
                        <input type="hidden" name="_csrf" value="<?= e(csrf_token()) ?>">
                        <input type="hidden" name="action" value="reject_manual">
                        <input type="hidden" name="sub_id" value="<?= (int)$r['id'] ?>">
                        <button class="btn btn-outline-danger btn-sm" onclick="return confirm('Reprovar este pagamento manual?')">
                          <i class="fas fa-xmark"></i> Reprovar
                        </button>
                      </form>
                    <?php else: ?>
                      <span class="text-secondary">-</span>
                    <?php endif; ?>
                  </td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
      </div>

      <small class="text-secondary d-block mt-2">Exibindo no máximo 500 registros por página.</small>
    </div>
  </section>
</div>
<?php include __DIR__ . '/../partials/footer.php'; ?>
