<?php
declare(strict_types=1);
require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../app/helpers.php';
require_admin();

if (is_post()) {
    if (!csrf_check($_POST['_csrf'] ?? null)) {
        flash('error', 'CSRF inválido');
        redirect('/admin/plans.php');
    }

    $id = (int)($_POST['id'] ?? 0);
    $data = [
      'name' => trim($_POST['name'] ?? ''),
      'slug' => trim($_POST['slug'] ?? ''),
      'description' => trim($_POST['description'] ?? ''),
      'price' => (float)($_POST['price'] ?? 0),
      'currency' => strtoupper(trim($_POST['currency'] ?? 'BRL')),
      'interval_unit' => $_POST['interval_unit'] ?? 'month',
      'interval_count' => max(1, (int)($_POST['interval_count'] ?? 1)),
      'trial_days' => max(0, (int)($_POST['trial_days'] ?? 0)),
      'is_enabled' => isset($_POST['is_enabled']) ? 1 : 0,
      'max_storage_gb' => max(1, (int)($_POST['max_storage_gb'] ?? 10)),
      'max_bandwidth_gb' => max(1, (int)($_POST['max_bandwidth_gb'] ?? 200)),
      'max_video_duration_min' => max(1, (int)($_POST['max_video_duration_min'] ?? 120)),
      'provider_plan_id_mp' => trim($_POST['provider_plan_id_mp'] ?? ''),
      'provider_plan_id_pg' => trim($_POST['provider_plan_id_pg'] ?? '')
    ];

    if ($data['name'] === '' || $data['slug'] === '') {
        flash('error', 'Nome e slug são obrigatórios.');
        redirect('/admin/plans.php');
    }

    if (!in_array($data['interval_unit'], ['day','week','month','year'], true)) {
        $data['interval_unit'] = 'month';
    }

    if ($id > 0) {
      $sql = "UPDATE plans SET
                name=:name, slug=:slug, description=:description, price=:price, currency=:currency,
                interval_unit=:interval_unit, interval_count=:interval_count, trial_days=:trial_days,
                is_enabled=:is_enabled, max_storage_gb=:max_storage_gb, max_bandwidth_gb=:max_bandwidth_gb,
                max_video_duration_min=:max_video_duration_min,
                provider_plan_id_mp=:provider_plan_id_mp, provider_plan_id_pg=:provider_plan_id_pg
              WHERE id=:id";
      $data['id'] = $id;
      db()->prepare($sql)->execute($data);
      flash('success', 'Plano atualizado.');
    } else {
      $sql = "INSERT INTO plans
              (name,slug,description,price,currency,interval_unit,interval_count,trial_days,is_enabled,max_storage_gb,max_bandwidth_gb,max_video_duration_min,provider_plan_id_mp,provider_plan_id_pg)
              VALUES
              (:name,:slug,:description,:price,:currency,:interval_unit,:interval_count,:trial_days,:is_enabled,:max_storage_gb,:max_bandwidth_gb,:max_video_duration_min,:provider_plan_id_mp,:provider_plan_id_pg)";
      db()->prepare($sql)->execute($data);
      flash('success', 'Plano criado.');
    }
    redirect('/admin/plans.php');
}

if (isset($_GET['toggle'])) {
    $id = (int)$_GET['toggle'];
    db()->prepare("UPDATE plans SET is_enabled = IF(is_enabled=1,0,1) WHERE id=:id")->execute(['id'=>$id]);
    flash('success', 'Status do plano atualizado.');
    redirect('/admin/plans.php');
}

$edit = null;
if (isset($_GET['edit'])) {
    $stmt = db()->prepare("SELECT * FROM plans WHERE id=:id");
    $stmt->execute(['id'=>(int)$_GET['edit']]);
    $edit = $stmt->fetch();
}

$plans = db()->query("SELECT * FROM plans ORDER BY id DESC")->fetchAll();

$pageTitle = 'Admin - Planos';
include __DIR__ . '/../partials/head.php';
include __DIR__ . '/../partials/navbar.php';
include __DIR__ . '/../partials/sidebar.php';
?>
<div class="content-wrapper">
  <div class="content-header"><div class="container-fluid"><h1>Gerenciar planos</h1></div></div>
  <section class="content"><div class="container-fluid">
    <?php include __DIR__ . '/../partials/flash.php'; ?>

    <div class="alert alert-info">
      <strong>Como preencher:</strong> os textos abaixo explicam para que serve cada campo do plano.
    </div>

    <div class="row">
      <div class="col-lg-6">
        <div class="card glass-card">
          <div class="card-header"><?= $edit ? 'Editar plano' : 'Novo plano' ?></div>
          <div class="card-body">
            <form method="post">
              <input type="hidden" name="_csrf" value="<?= e(csrf_token()) ?>">
              <input type="hidden" name="id" value="<?= (int)($edit['id'] ?? 0) ?>">

              <div class="mb-3">
                <label class="form-label">Nome do plano</label>
                <input class="form-control bg-transparent text-light" name="name" value="<?= e($edit['name'] ?? '') ?>" required>
                <small class="text-secondary">Nome exibido ao cliente no checkout e na área de assinatura.</small>
              </div>

              <div class="mb-3">
                <label class="form-label">Slug (identificador)</label>
                <input class="form-control bg-transparent text-light" name="slug" value="<?= e($edit['slug'] ?? '') ?>" required>
                <small class="text-secondary">Código único interno (ex.: <code>pro-mensal</code>). Não use espaços.</small>
              </div>

              <div class="mb-3">
                <label class="form-label">Descrição</label>
                <textarea class="form-control bg-transparent text-light" name="description" rows="2"><?= e($edit['description'] ?? '') ?></textarea>
                <small class="text-secondary">Resumo curto dos benefícios do plano.</small>
              </div>

              <div class="row g-2">
                <div class="col-md-6">
                  <label class="form-label">Valor (R$)</label>
                  <input class="form-control bg-transparent text-light" type="number" step="0.01" min="0" name="price" value="<?= e((string)($edit['price'] ?? '0')) ?>">
                  <small class="text-secondary">Preço cobrado por ciclo.</small>
                </div>
                <div class="col-md-6">
                  <label class="form-label">Moeda</label>
                  <input class="form-control bg-transparent text-light" name="currency" value="<?= e($edit['currency'] ?? 'BRL') ?>">
                  <small class="text-secondary">Use <code>BRL</code> para reais.</small>
                </div>

                <div class="col-md-6">
                  <label class="form-label">Frequência (unidade)</label>
                  <select class="form-select bg-transparent text-light" name="interval_unit">
                    <?php foreach (['day'=>'Dia','week'=>'Semana','month'=>'Mês','year'=>'Ano'] as $u => $ul): ?>
                      <option value="<?= e($u) ?>" <?= (($edit['interval_unit'] ?? 'month')===$u?'selected':'') ?>><?= e($ul) ?></option>
                    <?php endforeach; ?>
                  </select>
                  <small class="text-secondary">Periodicidade base da cobrança.</small>
                </div>

                <div class="col-md-6">
                  <label class="form-label">Frequência (quantidade)</label>
                  <input class="form-control bg-transparent text-light" type="number" min="1" name="interval_count" value="<?= e((string)($edit['interval_count'] ?? 1)) ?>">
                  <small class="text-secondary">Ex.: 1 mês, 3 meses, 1 ano.</small>
                </div>

                <div class="col-md-4">
                  <label class="form-label">Período de teste (dias)</label>
                  <input class="form-control bg-transparent text-light" type="number" min="0" name="trial_days" value="<?= e((string)($edit['trial_days'] ?? 0)) ?>">
                  <small class="text-secondary">Use 0 para desativar.</small>
                </div>

                <div class="col-md-4">
                  <label class="form-label">Limite de armazenamento (GB)</label>
                  <input class="form-control bg-transparent text-light" type="number" min="1" name="max_storage_gb" value="<?= e((string)($edit['max_storage_gb'] ?? 10)) ?>">
                  <small class="text-secondary">Espaço total de vídeos do cliente.</small>
                </div>

                <div class="col-md-4">
                  <label class="form-label">Limite de banda mensal (GB)</label>
                  <input class="form-control bg-transparent text-light" type="number" min="1" name="max_bandwidth_gb" value="<?= e((string)($edit['max_bandwidth_gb'] ?? 200)) ?>">
                  <small class="text-secondary">Tráfego mensal de reprodução.</small>
                </div>

                <div class="col-12">
                  <label class="form-label">Duração máxima por vídeo (min)</label>
                  <input class="form-control bg-transparent text-light" type="number" min="1" name="max_video_duration_min" value="<?= e((string)($edit['max_video_duration_min'] ?? 120)) ?>">
                  <small class="text-secondary">Tempo máximo permitido por upload.</small>
                </div>

                <div class="col-12">
                  <label class="form-label">ID do plano no Mercado Pago</label>
                  <input class="form-control bg-transparent text-light" name="provider_plan_id_mp" value="<?= e($edit['provider_plan_id_mp'] ?? '') ?>">
                  <small class="text-secondary">ID da recorrência criada no Mercado Pago (opcional).</small>
                </div>

                <div class="col-12">
                  <label class="form-label">ID do plano no PagBank</label>
                  <input class="form-control bg-transparent text-light" name="provider_plan_id_pg" value="<?= e($edit['provider_plan_id_pg'] ?? '') ?>">
                  <small class="text-secondary">ID da recorrência criada no PagBank (opcional).</small>
                </div>

                <div class="col-12 form-check mt-2 ms-1">
                  <input class="form-check-input" id="is_enabled" type="checkbox" name="is_enabled" <?= ((int)($edit['is_enabled'] ?? 1)===1?'checked':'') ?>>
                  <label class="form-check-label" for="is_enabled">Plano habilitado</label><br>
                  <small class="text-secondary">Se desativado, não aparece para novas assinaturas.</small>
                </div>
              </div>

              <button class="btn btn-neon mt-3"><?= $edit ? 'Atualizar' : 'Criar' ?></button>
              <?php if ($edit): ?><a class="btn btn-outline-light mt-3" href="<?= e(url('/admin/plans.php')) ?>">Cancelar</a><?php endif; ?>
            </form>
          </div>
        </div>
      </div>

      <div class="col-lg-6">
        <div class="card glass-card">
          <div class="card-body table-responsive">
            <table class="table table-dark table-hover">
              <thead><tr><th>ID</th><th>Nome</th><th>Preço</th><th>Intervalo</th><th>Status</th><th>Ações</th></tr></thead>
              <tbody>
                <?php foreach ($plans as $p): ?>
                <tr>
                  <td><?= (int)$p['id'] ?></td>
                  <td><?= e($p['name']) ?></td>
                  <td>R$ <?= number_format((float)$p['price'],2,',','.') ?></td>
                  <td><?= e(human_interval((string)$p['interval_unit'], (int)$p['interval_count'])) ?></td>
                  <td><?= (int)$p['is_enabled'] ? '<span class="badge bg-success">ON</span>' : '<span class="badge bg-secondary">OFF</span>' ?></td>
                  <td>
                    <a class="btn btn-sm btn-outline-info mb-1" href="<?= e(url('/admin/plans.php?edit=' . (int)$p['id'])) ?>">Editar</a>
                    <a class="btn btn-sm btn-outline-warning mb-1" href="<?= e(url('/admin/plans.php?toggle=' . (int)$p['id'])) ?>">Habilitar/Desabilitar</a>
                  </td>
                </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>
        </div>
      </div>
    </div>
  </div></section>
</div>
<?php include __DIR__ . '/../partials/footer.php'; ?>
