<?php
declare(strict_types=1);

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../app/helpers.php';

require_admin();

function ra_tail(?string $text, int $len = 220): string {
    $t = trim((string)$text);
    if ($t === '') return '';
    if (function_exists('mb_strlen') && function_exists('mb_substr')) {
        if (mb_strlen($t, 'UTF-8') <= $len) return $t;
        return mb_substr($t, 0, $len, 'UTF-8') . '…';
    }
    if (strlen($t) <= $len) return $t;
    return substr($t, 0, $len) . '…';
}

$quality = trim((string)($_GET['quality'] ?? ''));
$exitCodeRaw = trim((string)($_GET['exit_code'] ?? ''));
$jobId = (int)($_GET['job_id'] ?? 0);
$userId = (int)($_GET['user_id'] ?? 0);
$days = (int)($_GET['days'] ?? 0);
$q = trim((string)($_GET['q'] ?? ''));
$page = max(1, (int)($_GET['page'] ?? 1));
$perPage = max(20, min(200, (int)($_GET['per_page'] ?? 50)));
$export = (string)($_GET['export'] ?? '');

$hasExitCode = ($exitCodeRaw !== '' && preg_match('/^-?\d+$/', $exitCodeRaw) === 1);
$exitCode = $hasExitCode ? (int)$exitCodeRaw : null;

$where = [];
$params = [];

if ($quality !== '') {
    $where[] = 'l.quality = :quality';
    $params['quality'] = $quality;
}
if ($hasExitCode) {
    $where[] = 'l.exit_code = :exit_code';
    $params['exit_code'] = $exitCode;
}
if ($jobId > 0) {
    $where[] = 'l.video_job_id = :job_id';
    $params['job_id'] = $jobId;
}
if ($userId > 0) {
    $where[] = 'u.id = :user_id';
    $params['user_id'] = $userId;
}
if ($days === 7 || $days === 30) {
    $where[] = "l.created_at >= DATE_SUB(NOW(), INTERVAL :days DAY)";
    $params['days'] = $days;
}
if ($q !== '') {
    $where[] = '(v.title LIKE :q OR u.name LIKE :q OR u.email LIKE :q OR l.error_tail LIKE :q)';
    $params['q'] = '%' . $q . '%';
}

$whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

$fromSql = "
    FROM video_retry_attempt_logs l
    LEFT JOIN video_jobs j ON j.id = l.video_job_id
    LEFT JOIN videos v ON v.id = j.video_id
    LEFT JOIN users u ON u.id = v.user_id
";

$qualityOptions = db()->query("SELECT DISTINCT quality FROM video_retry_attempt_logs ORDER BY quality ASC")->fetchAll(PDO::FETCH_COLUMN) ?: [];
$exitOptions = db()->query("SELECT DISTINCT exit_code FROM video_retry_attempt_logs ORDER BY exit_code ASC")->fetchAll(PDO::FETCH_COLUMN) ?: [];
$userOptions = db()->query("
    SELECT DISTINCT u.id, u.name, u.email
    FROM video_retry_attempt_logs l
    LEFT JOIN video_jobs j ON j.id = l.video_job_id
    LEFT JOIN videos v ON v.id = j.video_id
    LEFT JOIN users u ON u.id = v.user_id
    WHERE u.id IS NOT NULL
    ORDER BY u.name ASC, u.email ASC
")->fetchAll(PDO::FETCH_ASSOC) ?: [];

$summarySql = "
SELECT 
  COUNT(*) AS total,
  SUM(CASE WHEN l.exit_code = 0 THEN 1 ELSE 0 END) AS ok_count,
  SUM(CASE WHEN l.exit_code <> 0 THEN 1 ELSE 0 END) AS fail_count,
  SUM(CASE WHEN l.quality='1080p' THEN 1 ELSE 0 END) AS q1080,
  SUM(CASE WHEN l.quality='720p' THEN 1 ELSE 0 END) AS q720,
  SUM(CASE WHEN l.quality='360p' THEN 1 ELSE 0 END) AS q360
{$fromSql}
{$whereSql}
";
$stSum = db()->prepare($summarySql);
$stSum->execute($params);
$summary = $stSum->fetch() ?: ['total'=>0,'ok_count'=>0,'fail_count'=>0,'q1080'=>0,'q720'=>0,'q360'=>0];

$countSql = "SELECT COUNT(*) {$fromSql} {$whereSql}";
$stCount = db()->prepare($countSql);
$stCount->execute($params);
$totalRows = (int)$stCount->fetchColumn();
$totalPages = max(1, (int)ceil($totalRows / $perPage));
$page = min($page, $totalPages);
$offset = ($page - 1) * $perPage;

$selectSql = "
SELECT
  l.id, l.video_job_id, l.quality, l.attempt_no, l.exit_code, l.error_tail, l.created_at, l.profile_json,
  j.status AS job_status, j.video_id,
  v.title AS video_title, v.status AS video_status,
  u.id AS user_id, u.name AS user_name, u.email AS user_email
{$fromSql}
{$whereSql}
ORDER BY l.id DESC
LIMIT {$perPage} OFFSET {$offset}
";
$stList = db()->prepare($selectSql);
$stList->execute($params);
$rows = $stList->fetchAll(PDO::FETCH_ASSOC) ?: [];

if ($export === 'csv') {
    header('Content-Type: text/csv; charset=UTF-8');
    header('Content-Disposition: attachment; filename="retry_attempts_' . date('Ymd_His') . '.csv"');

    $out = fopen('php://output', 'w');
    fwrite($out, "\xEF\xBB\xBF"); // BOM UTF-8
    fputcsv($out, ['id','created_at','video_job_id','video_id','quality','attempt_no','exit_code','job_status','video_status','video_title','user_id','user_name','user_email','error_tail']);

    $csvSql = "
    SELECT
      l.id, l.created_at, l.video_job_id, j.video_id, l.quality, l.attempt_no, l.exit_code,
      j.status AS job_status, v.status AS video_status, v.title AS video_title,
      u.id AS user_id, u.name AS user_name, u.email AS user_email, l.error_tail
    {$fromSql}
    {$whereSql}
    ORDER BY l.id DESC
    LIMIT 5000
    ";
    $stCsv = db()->prepare($csvSql);
    $stCsv->execute($params);
    while ($r = $stCsv->fetch(PDO::FETCH_ASSOC)) {
        fputcsv($out, [
            $r['id'] ?? '',
            $r['created_at'] ?? '',
            $r['video_job_id'] ?? '',
            $r['video_id'] ?? '',
            $r['quality'] ?? '',
            $r['attempt_no'] ?? '',
            $r['exit_code'] ?? '',
            $r['job_status'] ?? '',
            $r['video_status'] ?? '',
            $r['video_title'] ?? '',
            $r['user_id'] ?? '',
            $r['user_name'] ?? '',
            $r['user_email'] ?? '',
            preg_replace('/\s+/', ' ', (string)($r['error_tail'] ?? ''))
        ]);
    }
    fclose($out);
    exit;
}

$paramsBase = $_GET;
unset($paramsBase['page'], $paramsBase['export']);
$qsBase = http_build_query($paramsBase);

$pageTitle = 'Admin - Histórico de Tentativas FFmpeg';
include __DIR__ . '/../partials/head.php';
include __DIR__ . '/../partials/navbar.php';
include __DIR__ . '/../partials/sidebar.php';
?>
<div class="content-wrapper">
  <div class="content-header">
    <div class="container-fluid d-flex justify-content-between align-items-center flex-wrap">
      <h1 class="mb-2"><i class="fa-solid fa-wave-square mr-2"></i>Histórico de tentativas FFmpeg</h1>
      <div class="mb-2">
        <a class="btn btn-outline-light btn-sm" href="<?= e(url('/admin/system_validation.php')) ?>">
          <i class="fa-solid fa-shield-halved"></i> Validação do Sistema
        </a>
        <a class="btn btn-primary btn-sm" href="<?= e(url('/admin/retry_attempts.php?' . ($qsBase ? $qsBase . '&' : '') . 'export=csv')) ?>">
          <i class="fa-solid fa-file-csv"></i> Exportar CSV
        </a>
      </div>
    </div>
  </div>

  <section class="content">
    <div class="container-fluid">

      <div class="row mb-3">
        <div class="col-md-2 col-6"><div class="small-box bg-dark border"><div class="inner"><h3><?= (int)$summary['total'] ?></h3><p>Total de tentativas</p></div><div class="icon"><i class="fa-solid fa-list"></i></div></div></div>
        <div class="col-md-2 col-6"><div class="small-box bg-success"><div class="inner"><h3><?= (int)$summary['ok_count'] ?></h3><p>Exit code 0</p></div><div class="icon"><i class="fa-solid fa-circle-check"></i></div></div></div>
        <div class="col-md-2 col-6"><div class="small-box bg-danger"><div class="inner"><h3><?= (int)$summary['fail_count'] ?></h3><p>Com falha</p></div><div class="icon"><i class="fa-solid fa-triangle-exclamation"></i></div></div></div>
        <div class="col-md-2 col-6"><div class="small-box bg-info"><div class="inner"><h3><?= (int)$summary['q1080'] ?></h3><p>1080p</p></div><div class="icon"><i class="fa-solid fa-tv"></i></div></div></div>
        <div class="col-md-2 col-6"><div class="small-box bg-info"><div class="inner"><h3><?= (int)$summary['q720'] ?></h3><p>720p</p></div><div class="icon"><i class="fa-solid fa-film"></i></div></div></div>
        <div class="col-md-2 col-6"><div class="small-box bg-info"><div class="inner"><h3><?= (int)$summary['q360'] ?></h3><p>360p</p></div><div class="icon"><i class="fa-solid fa-mobile-screen"></i></div></div></div>
      </div>

      <div class="card card-outline card-primary">
        <div class="card-header"><strong>Filtros</strong></div>
        <div class="card-body">
          <form method="get" class="row g-2">
            <div class="col-md-2">
              <label class="form-label">Qualidade</label>
              <select name="quality" class="form-control form-control-sm">
                <option value="">Todas</option>
                <?php foreach ($qualityOptions as $qOpt): ?>
                  <option value="<?= e((string)$qOpt) ?>" <?= ($quality === (string)$qOpt ? 'selected' : '') ?>><?= e((string)$qOpt) ?></option>
                <?php endforeach; ?>
              </select>
            </div>
            <div class="col-md-2">
              <label class="form-label">Exit Code</label>
              <select name="exit_code" class="form-control form-control-sm">
                <option value="">Todos</option>
                <?php foreach ($exitOptions as $ec): ?>
                  <option value="<?= e((string)$ec) ?>" <?= ($exitCodeRaw !== '' && (string)$ec === $exitCodeRaw ? 'selected' : '') ?>><?= e((string)$ec) ?></option>
                <?php endforeach; ?>
              </select>
            </div>
            <div class="col-md-2">
              <label class="form-label">Período</label>
              <select name="days" class="form-control form-control-sm">
                <option value="0" <?= ($days === 0 ? 'selected' : '') ?>>Todo histórico</option>
                <option value="7" <?= ($days === 7 ? 'selected' : '') ?>>Últimos 7 dias</option>
                <option value="30" <?= ($days === 30 ? 'selected' : '') ?>>Últimos 30 dias</option>
              </select>
            </div>
            <div class="col-md-2">
              <label class="form-label">Job ID</label>
              <input type="number" name="job_id" class="form-control form-control-sm" value="<?= $jobId > 0 ? (int)$jobId : '' ?>" placeholder="Ex.: 154">
            </div>
            <div class="col-md-2">
              <label class="form-label">Usuário</label>
              <select name="user_id" class="form-control form-control-sm">
                <option value="0">Todos</option>
                <?php foreach ($userOptions as $uopt): ?>
                  <?php $uid = (int)($uopt['id'] ?? 0); ?>
                  <option value="<?= $uid ?>" <?= ($uid === $userId ? 'selected' : '') ?>>
                    #<?= $uid ?> - <?= e((string)($uopt['name'] ?: $uopt['email'])) ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>
            <div class="col-md-2">
              <label class="form-label">Busca</label>
              <input type="text" name="q" class="form-control form-control-sm" value="<?= e($q) ?>" placeholder="título, usuário, erro...">
            </div>

            <div class="col-md-2">
              <label class="form-label">Por página</label>
              <select name="per_page" class="form-control form-control-sm">
                <?php foreach ([20,50,100,200] as $pp): ?>
                  <option value="<?= $pp ?>" <?= ($perPage === $pp ? 'selected' : '') ?>><?= $pp ?></option>
                <?php endforeach; ?>
              </select>
            </div>
            <div class="col-md-10 d-flex align-items-end">
              <button class="btn btn-primary btn-sm mr-2" type="submit"><i class="fa-solid fa-filter"></i> Filtrar</button>
              <a class="btn btn-outline-secondary btn-sm" href="<?= e(url('/admin/retry_attempts.php')) ?>"><i class="fa-solid fa-eraser"></i> Limpar</a>
            </div>
          </form>
        </div>
      </div>

      <div class="card">
        <div class="card-header d-flex justify-content-between">
          <strong>Resultados</strong>
          <span class="text-muted">Total: <?= (int)$totalRows ?> registro(s)</span>
        </div>
        <div class="card-body table-responsive p-0">
          <table class="table table-sm table-hover align-middle mb-0">
            <thead>
              <tr>
                <th>#</th>
                <th>Data</th>
                <th>Job</th>
                <th>Qualidade</th>
                <th>Tentativa</th>
                <th>Exit</th>
                <th>Vídeo</th>
                <th>Usuário</th>
                <th>Erro (tail)</th>
              </tr>
            </thead>
            <tbody>
            <?php if (!$rows): ?>
              <tr><td colspan="9" class="text-center text-muted py-4">Nenhum registro encontrado para os filtros informados.</td></tr>
            <?php else: foreach ($rows as $r): ?>
              <tr>
                <td><?= (int)$r['id'] ?></td>
                <td><?= e((string)$r['created_at']) ?></td>
                <td>
                  <a href="<?= e(url('/admin/video_jobs.php')) ?>" class="text-info">#<?= (int)$r['video_job_id'] ?></a>
                  <div class="small text-muted">status: <?= e((string)($r['job_status'] ?? '-')) ?></div>
                </td>
                <td><span class="badge badge-info"><?= e((string)($r['quality'] ?? '-')) ?></span></td>
                <td><?= (int)($r['attempt_no'] ?? 0) ?></td>
                <td>
                  <?php $exit = (int)($r['exit_code'] ?? 0); ?>
                  <?php if ($exit === 0): ?>
                    <span class="badge badge-success">0</span>
                  <?php else: ?>
                    <span class="badge badge-danger"><?= $exit ?></span>
                  <?php endif; ?>
                </td>
                <td>
                  <div><?= e((string)($r['video_title'] ?? '—')) ?></div>
                  <div class="small text-muted">video_id: <?= (int)($r['video_id'] ?? 0) ?> | status: <?= e((string)($r['video_status'] ?? '-')) ?></div>
                </td>
                <td>
                  <div><?= e((string)($r['user_name'] ?? '—')) ?></div>
                  <div class="small text-muted"><?= e((string)($r['user_email'] ?? '')) ?></div>
                </td>
                <td style="max-width:420px;">
                  <code class="text-wrap d-block" style="white-space:pre-wrap;"><?= e(ra_tail((string)($r['error_tail'] ?? ''), 300)) ?></code>
                </td>
              </tr>
            <?php endforeach; endif; ?>
            </tbody>
          </table>
        </div>

        <?php if ($totalPages > 1): ?>
        <div class="card-footer">
          <nav>
            <ul class="pagination pagination-sm m-0">
              <?php
                $buildPageUrl = function(int $p) use ($paramsBase): string {
                    $params = $paramsBase;
                    $params['page'] = $p;
                    return url('/admin/retry_attempts.php?' . http_build_query($params));
                };
              ?>
              <li class="page-item <?= $page <= 1 ? 'disabled' : '' ?>">
                <a class="page-link" href="<?= e($buildPageUrl(max(1, $page - 1))) ?>">&laquo;</a>
              </li>
              <?php
                $start = max(1, $page - 3);
                $end = min($totalPages, $page + 3);
                for ($p = $start; $p <= $end; $p++):
              ?>
                <li class="page-item <?= $p === $page ? 'active' : '' ?>">
                  <a class="page-link" href="<?= e($buildPageUrl($p)) ?>"><?= $p ?></a>
                </li>
              <?php endfor; ?>
              <li class="page-item <?= $page >= $totalPages ? 'disabled' : '' ?>">
                <a class="page-link" href="<?= e($buildPageUrl(min($totalPages, $page + 1))) ?>">&raquo;</a>
              </li>
            </ul>
          </nav>
        </div>
        <?php endif; ?>
      </div>

    </div>
  </section>
</div>
<?php include __DIR__ . '/../partials/footer.php'; ?>
