<?php
/**
 * System Validation (DB + Pagamentos + FFmpeg + Permissões)
 * Compatível com instalação em subpasta no XAMPP.
 */
declare(strict_types=1);

session_start();

// --- proteção admin ---
$projectRoot = dirname(__DIR__, 2);
$configFile  = $projectRoot . '/config/config.php';
if (file_exists($configFile)) {
    require_once $configFile;
}

$user = $_SESSION['user'] ?? null;
if (is_array($user)) {
    $role = $user['role'] ?? $user['perfil'] ?? 'client';
    if ($role !== 'admin') {
        http_response_code(403);
        echo 'Acesso negado. Somente admin.';
        exit;
    }
}

function h(?string $v): string { return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }

function loadEnvFile(string $envPath): array {
    $out = [];
    if (!is_file($envPath)) return $out;
    $lines = file($envPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    if (!$lines) return $out;

    foreach ($lines as $line) {
        $line = trim($line);
        if ($line === '' || str_starts_with($line, '#')) continue;
        $eq = strpos($line, '=');
        if ($eq === false) continue;
        $k = trim(substr($line, 0, $eq));
        $v = trim(substr($line, $eq + 1));
        if ($v !== '' && (($v[0] === '"' && str_ends_with($v, '"')) || ($v[0] === "'" && str_ends_with($v, "'")))) {
            $v = substr($v, 1, -1);
        }
        $out[$k] = $v;
    }
    return $out;
}

function envGet(array $env, string $key, ?string $default = null): ?string {
    $v = $env[$key] ?? getenv($key);
    if ($v === false || $v === null || $v === '') return $default;
    return (string)$v;
}

function statusBadge(bool $ok, string $okText = 'OK', string $failText = 'Falha'): string {
    $cls = $ok ? 'badge-success' : 'badge-danger';
    $txt = $ok ? $okText : $failText;
    return "<span class='badge {$cls}'>{$txt}</span>";
}

function checkBinary(?string $bin, string $fallbackCmd): array {
    $resolved = null;
    if ($bin && $bin !== '') {
        $resolved = $bin;
    } else {
        $where = [];
        $ret = 1;
        @exec($fallbackCmd . ' 2>NUL', $where, $ret);
        if ($ret === 0 && !empty($where[0])) {
            $resolved = trim($where[0]);
        }
    }

    if (!$resolved) return [false, 'Não encontrado', null];

    $out = [];
    $ret = 1;
    @exec('"' . $resolved . '" -version 2>&1', $out, $ret);
    if ($ret !== 0) return [false, 'Encontrado, mas não executa', $resolved];

    $first = $out[0] ?? 'Executável válido';
    return [true, $first, $resolved];
}

function tableExists(PDO $pdo, string $dbName, string $table): bool {
    $sql = "SELECT 1 FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = :db AND TABLE_NAME = :tb LIMIT 1";
    $st = $pdo->prepare($sql);
    $st->execute([':db' => $dbName, ':tb' => $table]);
    $row = $st->fetch(PDO::FETCH_NUM);
    $st->closeCursor();
    return (bool)$row;
}

function getTableColumns(PDO $pdo, string $table): array {
    $cols = [];
    $st = $pdo->query("SHOW COLUMNS FROM `" . str_replace('`', '``', $table) . "`");
    $rows = $st->fetchAll(PDO::FETCH_ASSOC);
    $st->closeCursor();
    foreach ($rows as $r) {
        $f = (string)($r['Field'] ?? '');
        if ($f !== '') $cols[] = $f;
    }
    return $cols;
}

function getPaymentSettings(PDO $pdo): array {
    $exists = false;
    try {
        $st = $pdo->query("SHOW TABLES LIKE 'payment_settings'");
        $exists = (bool)$st->fetch(PDO::FETCH_NUM);
        $st->closeCursor();
    } catch (Throwable $e) {
        $exists = false;
    }
    if (!$exists) return [];

    try {
        $rows = $pdo->query("SELECT provider, setting_key, setting_value FROM payment_settings")->fetchAll(PDO::FETCH_ASSOC);
    } catch (Throwable $e) {
        return [];
    }

    $out = [];
    foreach ($rows as $r) {
        $p = (string)($r['provider'] ?? '');
        $k = (string)($r['setting_key'] ?? '');
        $v = (string)($r['setting_value'] ?? '');
        if ($p === '' || $k === '') continue;
        if (!isset($out[$p])) $out[$p] = [];
        $out[$p][$k] = $v;
    }
    return $out;
}

function ensureWritableDir(string $path): array {
    $steps = [];
    $ok = true;

    if (!is_dir($path)) {
        if (@mkdir($path, 0777, true)) {
            $steps[] = 'Pasta criada';
        } else {
            $steps[] = 'Falha ao criar pasta';
            $ok = false;
        }
    } else {
        $steps[] = 'Pasta já existe';
    }

    if (!@is_writable($path)) {
        @chmod($path, 0777); // no-op em alguns ambientes Windows
        if (@is_writable($path)) {
            $steps[] = 'Permissão ajustada';
        } else {
            $steps[] = 'Ainda sem escrita (ajuste ACL manual no Windows)';
            $ok = false;
        }
    } else {
        $steps[] = 'Já possui escrita';
    }

    $testFile = rtrim($path, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . '.__write_test_' . bin2hex(random_bytes(4));
    $written = @file_put_contents($testFile, 'ok');
    if ($written !== false) {
        @unlink($testFile);
        $steps[] = 'Teste de escrita OK';
    } else {
        $steps[] = 'Teste de escrita falhou';
        $ok = false;
    }

    return [$ok, implode(' | ', $steps)];
}

$env = loadEnvFile($projectRoot . '/.env');

$paths = [
    'public/uploads' => $projectRoot . '/public/uploads',
    'storage/hls'    => $projectRoot . '/storage/hls',
    'storage/logs'   => $projectRoot . '/storage/logs',
];

$requestedAction = $_SERVER['REQUEST_METHOD'] === 'POST' ? (string)($_POST['action'] ?? '') : '';

$permFixFeedback = [];
if ($requestedAction === 'fix_permissions') {
    foreach ($paths as $label => $abs) {
        [$okFix, $msgFix] = ensureWritableDir($abs);
        $permFixFeedback[$label] = ['ok' => $okFix, 'msg' => $msgFix];
    }
}

$jobFixFeedback = null;
$jobFixByUserFeedback = null;
$jobFixByPlanFeedback = null;
$jobFixByPlanUserFeedback = null;
$jobFixFfmpegFeedback = null;
$selectedUserId = (int)($_POST['user_id'] ?? ($_GET['user_id'] ?? 0));
$selectedPlanId = (int)($_POST['plan_id'] ?? ($_GET['plan_id'] ?? 0));
$selectedPeriod = (string)($_POST['period'] ?? ($_GET['period'] ?? 'all'));
if (!in_array($selectedPeriod, ['all','7','30'], true)) {
    $selectedPeriod = 'all';
}
$selectedPeriodDays = $selectedPeriod === '7' ? 7 : ($selectedPeriod === '30' ? 30 : 0);
$selectedTrendPeriod = (string)($_POST['trend_period'] ?? ($_GET['trend_period'] ?? '7'));
if (!in_array($selectedTrendPeriod, ['7','30'], true)) {
    $selectedTrendPeriod = '7';
}
$selectedTrendDays = (int)$selectedTrendPeriod;
$errorTrendSeries = [];
$maxTrendTotal = 1;
$usersWithFailedJobs = [];
$plansForFilter = [];
$usersHasCurrentPlan = false;
$jobHasCreatedAt = false;

$checks = [];
$details = [];

$jobErrorColumn = null;
$errorTypeCounts = [
    'ffmpeg' => 0,
    'storage' => 0,
    'network' => 0,
    'db' => 0,
    'webhook' => 0,
    'other' => 0,
];
$totalFailedJobs = 0;


// --- DB ---
$dbHost = envGet($env, 'DB_HOST', '127.0.0.1');
$dbPort = envGet($env, 'DB_PORT', '3306');
$dbName = envGet($env, 'DB_DATABASE', 'video_saas');
$dbUser = envGet($env, 'DB_USERNAME', 'root');
$dbPass = envGet($env, 'DB_PASSWORD', '');

$pdo = null;
$dbError = null;
try {
    $dsn = "mysql:host={$dbHost};port={$dbPort};dbname={$dbName};charset=utf8mb4";
    $pdo = new PDO($dsn, $dbUser, $dbPass, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::MYSQL_ATTR_USE_BUFFERED_QUERY => true,
    ]);
    $checks['db_connection'] = true;
} catch (Throwable $e) {
    $checks['db_connection'] = false;
    $dbError = $e->getMessage();
}

$requiredTables = [
    'users',
    'plans',
    'subscriptions',
    'videos',
    'video_jobs',
    'payment_settings',
    'manual_payment_proofs',
];

if ($pdo) {
    foreach ($requiredTables as $tb) {
        $checks['table_' . $tb] = tableExists($pdo, $dbName, $tb);
    }

    if (($checks['table_users'] ?? false)) {
        try {
            $uCols = getTableColumns($pdo, 'users');
            $usersHasCurrentPlan = in_array('current_plan_id', $uCols, true);
        } catch (Throwable $e) {
            $usersHasCurrentPlan = false;
        }
    }
    if (($checks['table_video_jobs'] ?? false)) {
        try {
            $jCols = getTableColumns($pdo, 'video_jobs');
            $jobHasCreatedAt = in_array('created_at', $jCols, true);
            if (in_array('error_message', $jCols, true)) {
                $jobErrorColumn = 'error_message';
            } elseif (in_array('last_error', $jCols, true)) {
                $jobErrorColumn = 'last_error';
            }
        } catch (Throwable $e) {
            $jobHasCreatedAt = false;
            $jobErrorColumn = null;
        }
    }

    if ($requestedAction === 'reprocess_failed_jobs') {
        if (!($checks['table_video_jobs'] ?? false)) {
            $jobFixFeedback = ['ok' => false, 'msg' => 'Tabela video_jobs não encontrada.'];
        } else {
            try {
                $cols = getTableColumns($pdo, 'video_jobs');
                $set = [
                    "status='queued'",
                    "attempts=0",
                ];
                if (in_array('run_after', $cols, true)) {
                    $set[] = "run_after=NOW()";
                }
                if (in_array('started_at', $cols, true)) {
                    $set[] = "started_at=NULL";
                }
                if (in_array('finished_at', $cols, true)) {
                    $set[] = "finished_at=NULL";
                }
                if (in_array('error_message', $cols, true)) {
                    $set[] = "error_message=NULL";
                } elseif (in_array('last_error', $cols, true)) {
                    $set[] = "last_error=NULL";
                }
                if (in_array('updated_at', $cols, true)) {
                    $set[] = "updated_at=NOW()";
                }

                $sql = "UPDATE video_jobs SET " . implode(', ', $set) . " WHERE status='failed'";
                $affected = $pdo->exec($sql);
                $jobFixFeedback = ['ok' => true, 'msg' => 'Jobs reprocessados: ' . (int)$affected . '.'];
            } catch (Throwable $e) {
                $jobFixFeedback = ['ok' => false, 'msg' => 'Falha ao reprocessar: ' . $e->getMessage()];
            }
        }
    }


if ($requestedAction === 'reprocess_failed_jobs_ffmpeg_only') {
    if (!($checks['table_video_jobs'] ?? false)) {
        $jobFixFfmpegFeedback = ['ok' => false, 'msg' => 'Tabela video_jobs não encontrada.'];
    } elseif (!($jobErrorColumn)) {
        $jobFixFfmpegFeedback = ['ok' => false, 'msg' => 'Coluna de erro (error_message/last_error) não encontrada em video_jobs.'];
    } else {
        try {
            $cols = getTableColumns($pdo, 'video_jobs');
            $set = [
                "status='queued'",
                "attempts=0",
            ];
            if (in_array('run_after', $cols, true)) {
                $set[] = "run_after=NOW()";
            }
            if (in_array('started_at', $cols, true)) {
                $set[] = "started_at=NULL";
            }
            if (in_array('finished_at', $cols, true)) {
                $set[] = "finished_at=NULL";
            }
            if (in_array('error_message', $cols, true)) {
                $set[] = "error_message=NULL";
            } elseif (in_array('last_error', $cols, true)) {
                $set[] = "last_error=NULL";
            }
            if (in_array('updated_at', $cols, true)) {
                $set[] = "updated_at=NOW()";
            }

            $where = "status='failed' AND (LOWER(COALESCE($jobErrorColumn,'')) LIKE '%ffmpeg%' OR LOWER(COALESCE($jobErrorColumn,'')) LIKE '%ffprobe%')";
            if ($selectedPeriodDays > 0 && $jobHasCreatedAt) {
                $where .= " AND created_at >= DATE_SUB(NOW(), INTERVAL " . (int)$selectedPeriodDays . " DAY)";
            }

            $sql = "UPDATE video_jobs SET " . implode(', ', $set) . " WHERE " . $where;
            $affected = $pdo->exec($sql);
            $suf = ($selectedPeriodDays > 0 && $jobHasCreatedAt) ? ' (período: ' . $selectedPeriodDays . 'd)' : '';
            $jobFixFfmpegFeedback = ['ok' => true, 'msg' => 'Jobs FFmpeg/FFprobe reprocessados: ' . (int)$affected . '.' . $suf];
        } catch (Throwable $e) {
            $jobFixFfmpegFeedback = ['ok' => false, 'msg' => 'Falha ao reprocessar somente FFmpeg: ' . $e->getMessage()];
        }
    }
}



    if ($requestedAction === 'reprocess_failed_jobs_user') {
        $selectedUserId = (int)($_POST['user_id'] ?? 0);

        if ($selectedUserId <= 0) {
            $jobFixByUserFeedback = ['ok' => false, 'msg' => 'Selecione um usuário válido.'];
        } elseif (!($checks['table_video_jobs'] ?? false) || !($checks['table_videos'] ?? false)) {
            $jobFixByUserFeedback = ['ok' => false, 'msg' => 'Tabelas video_jobs/videos não encontradas.'];
        } else {
            try {
                $cols = getTableColumns($pdo, 'video_jobs');
                $set = [
                    "j.status='queued'",
                    "j.attempts=0",
                ];
                if (in_array('run_after', $cols, true)) {
                    $set[] = "j.run_after=NOW()";
                }
                if (in_array('started_at', $cols, true)) {
                    $set[] = "j.started_at=NULL";
                }
                if (in_array('finished_at', $cols, true)) {
                    $set[] = "j.finished_at=NULL";
                }
                if (in_array('error_message', $cols, true)) {
                    $set[] = "j.error_message=NULL";
                } elseif (in_array('last_error', $cols, true)) {
                    $set[] = "j.last_error=NULL";
                }
                if (in_array('updated_at', $cols, true)) {
                    $set[] = "j.updated_at=NOW()";
                }

                $params = [':uid' => $selectedUserId];
                $extra = '';

                if ($selectedPlanId > 0 && $usersHasCurrentPlan) {
                    $extra .= " AND u.current_plan_id = :plan_id";
                    $params[':plan_id'] = $selectedPlanId;
                }

                if ($selectedPeriodDays > 0 && $jobHasCreatedAt) {
                    $extra .= " AND j.created_at >= DATE_SUB(NOW(), INTERVAL " . (int)$selectedPeriodDays . " DAY)";
                }

                $sql = "UPDATE video_jobs j "
                     . "INNER JOIN videos v ON v.id = j.video_id "
                     . "INNER JOIN users u ON u.id = v.user_id "
                     . "SET " . implode(', ', $set) . " "
                     . "WHERE j.status='failed' AND v.user_id = :uid" . $extra;

                $st = $pdo->prepare($sql);
                $st->execute($params);
                $affected = $st->rowCount();
                $st->closeCursor();

                $filtroTxt = [];
                if ($selectedPlanId > 0 && $usersHasCurrentPlan) $filtroTxt[] = 'plano';
                if ($selectedPeriodDays > 0 && $jobHasCreatedAt) $filtroTxt[] = 'período ' . $selectedPeriodDays . 'd';
                $suf = !empty($filtroTxt) ? ' (filtro: ' . implode(' + ', $filtroTxt) . ')' : '';

                $jobFixByUserFeedback = ['ok' => true, 'msg' => 'Jobs reprocessados para usuário ID ' . $selectedUserId . ': ' . (int)$affected . '.' . $suf];
            } catch (Throwable $e) {
                $jobFixByUserFeedback = ['ok' => false, 'msg' => 'Falha ao reprocessar por usuário: ' . $e->getMessage()];
            }
        }
    }



if ($requestedAction === 'reprocess_failed_jobs_plan_user') {
    $selectedUserId = (int)($_POST['user_id'] ?? 0);

    if ($selectedPlanId <= 0) {
        $jobFixByPlanUserFeedback = ['ok' => false, 'msg' => 'Selecione um plano válido.'];
    } elseif ($selectedUserId <= 0) {
        $jobFixByPlanUserFeedback = ['ok' => false, 'msg' => 'Selecione um usuário válido.'];
    } elseif (!$usersHasCurrentPlan) {
        $jobFixByPlanUserFeedback = ['ok' => false, 'msg' => 'A coluna users.current_plan_id não foi encontrada.'];
    } elseif (!($checks['table_video_jobs'] ?? false) || !($checks['table_videos'] ?? false)) {
        $jobFixByPlanUserFeedback = ['ok' => false, 'msg' => 'Tabelas video_jobs/videos não encontradas.'];
    } else {
        try {
            $cols = getTableColumns($pdo, 'video_jobs');
            $set = [
                "j.status='queued'",
                "j.attempts=0",
            ];
            if (in_array('run_after', $cols, true)) {
                $set[] = "j.run_after=NOW()";
            }
            if (in_array('started_at', $cols, true)) {
                $set[] = "j.started_at=NULL";
            }
            if (in_array('finished_at', $cols, true)) {
                $set[] = "j.finished_at=NULL";
            }
            if (in_array('error_message', $cols, true)) {
                $set[] = "j.error_message=NULL";
            } elseif (in_array('last_error', $cols, true)) {
                $set[] = "j.last_error=NULL";
            }
            if (in_array('updated_at', $cols, true)) {
                $set[] = "j.updated_at=NOW()";
            }

            $params = [
                ':uid' => $selectedUserId,
                ':plan_id' => $selectedPlanId,
            ];
            $extra = '';
            if ($selectedPeriodDays > 0 && $jobHasCreatedAt) {
                $extra .= " AND j.created_at >= DATE_SUB(NOW(), INTERVAL " . (int)$selectedPeriodDays . " DAY)";
            }

            $sql = "UPDATE video_jobs j "
                 . "INNER JOIN videos v ON v.id = j.video_id "
                 . "INNER JOIN users u ON u.id = v.user_id "
                 . "SET " . implode(', ', $set) . " "
                 . "WHERE j.status='failed' AND v.user_id = :uid AND u.current_plan_id = :plan_id" . $extra;

            $st = $pdo->prepare($sql);
            $st->execute($params);
            $affected = $st->rowCount();
            $st->closeCursor();

            $suf = ($selectedPeriodDays > 0 && $jobHasCreatedAt) ? ' (período: ' . $selectedPeriodDays . 'd)' : '';
            $jobFixByPlanUserFeedback = ['ok' => true, 'msg' => 'Jobs reprocessados para plano ID ' . $selectedPlanId . ' + usuário ID ' . $selectedUserId . ': ' . (int)$affected . '.' . $suf];
        } catch (Throwable $e) {
            $jobFixByPlanUserFeedback = ['ok' => false, 'msg' => 'Falha ao reprocessar por plano + usuário: ' . $e->getMessage()];
        }
    }
}


if ($requestedAction === 'reprocess_failed_jobs_plan') {
    if ($selectedPlanId <= 0) {
        $jobFixByPlanFeedback = ['ok' => false, 'msg' => 'Selecione um plano válido para reprocessar por plano.'];
    } elseif (!$usersHasCurrentPlan) {
        $jobFixByPlanFeedback = ['ok' => false, 'msg' => 'A coluna users.current_plan_id não foi encontrada.'];
    } elseif (!($checks['table_video_jobs'] ?? false) || !($checks['table_videos'] ?? false)) {
        $jobFixByPlanFeedback = ['ok' => false, 'msg' => 'Tabelas video_jobs/videos não encontradas.'];
    } else {
        try {
            $cols = getTableColumns($pdo, 'video_jobs');
            $set = [
                "j.status='queued'",
                "j.attempts=0",
            ];
            if (in_array('run_after', $cols, true)) {
                $set[] = "j.run_after=NOW()";
            }
            if (in_array('started_at', $cols, true)) {
                $set[] = "j.started_at=NULL";
            }
            if (in_array('finished_at', $cols, true)) {
                $set[] = "j.finished_at=NULL";
            }
            if (in_array('error_message', $cols, true)) {
                $set[] = "j.error_message=NULL";
            } elseif (in_array('last_error', $cols, true)) {
                $set[] = "j.last_error=NULL";
            }
            if (in_array('updated_at', $cols, true)) {
                $set[] = "j.updated_at=NOW()";
            }

            $params = [':plan_id' => $selectedPlanId];
            $extra = '';

            if ($selectedPeriodDays > 0 && $jobHasCreatedAt) {
                $extra .= " AND j.created_at >= DATE_SUB(NOW(), INTERVAL " . (int)$selectedPeriodDays . " DAY)";
            }

            $sql = "UPDATE video_jobs j "
                 . "INNER JOIN videos v ON v.id = j.video_id "
                 . "INNER JOIN users u ON u.id = v.user_id "
                 . "SET " . implode(', ', $set) . " "
                 . "WHERE j.status='failed' AND u.current_plan_id = :plan_id" . $extra;

            $st = $pdo->prepare($sql);
            $st->execute($params);
            $affected = $st->rowCount();
            $st->closeCursor();

            $suf = ($selectedPeriodDays > 0 && $jobHasCreatedAt) ? ' (período: ' . $selectedPeriodDays . 'd)' : '';
            $jobFixByPlanFeedback = ['ok' => true, 'msg' => 'Jobs reprocessados para plano ID ' . $selectedPlanId . ': ' . (int)$affected . '.' . $suf];
        } catch (Throwable $e) {
            $jobFixByPlanFeedback = ['ok' => false, 'msg' => 'Falha ao reprocessar por plano: ' . $e->getMessage()];
        }
    }
}


    try {
        if ($checks['table_plans'] ?? false) {
            $stPlans = $pdo->query("SELECT id, name, slug FROM plans ORDER BY name ASC, id ASC");
            $plansForFilter = $stPlans->fetchAll(PDO::FETCH_ASSOC);
            $stPlans->closeCursor();
        }
    } catch (Throwable $e) {
        $plansForFilter = [];
    }

    try {
        $sqlUsersFailed = "SELECT u.id, u.name, u.email, u.current_plan_id, p.name AS plan_name, COUNT(*) AS failed_count
"
                       . "FROM video_jobs j
"
                       . "INNER JOIN videos v ON v.id = j.video_id
"
                       . "INNER JOIN users u ON u.id = v.user_id
"
                       . "LEFT JOIN plans p ON p.id = u.current_plan_id
"
                       . "WHERE j.status = 'failed'";

        $paramsUF = [];
        if ($selectedPlanId > 0 && $usersHasCurrentPlan) {
            $sqlUsersFailed .= " AND u.current_plan_id = :plan_id";
            $paramsUF[':plan_id'] = $selectedPlanId;
        }
        if ($selectedPeriodDays > 0 && $jobHasCreatedAt) {
            $sqlUsersFailed .= " AND j.created_at >= DATE_SUB(NOW(), INTERVAL " . (int)$selectedPeriodDays . " DAY)";
        }

        $sqlUsersFailed .= "
GROUP BY u.id, u.name, u.email, u.current_plan_id, p.name
"
                        . "ORDER BY failed_count DESC, u.id DESC
"
                        . "LIMIT 300";

        $stUF = $pdo->prepare($sqlUsersFailed);
        $stUF->execute($paramsUF);
        $usersWithFailedJobs = $stUF->fetchAll(PDO::FETCH_ASSOC);
        $stUF->closeCursor();
    } catch (Throwable $e) {
        $usersWithFailedJobs = [];
    }


// Contadores por tipo de erro (somente jobs com status=failed)
try {
    if (($checks['table_video_jobs'] ?? false)) {
        $totalSql = "SELECT COUNT(*) FROM video_jobs WHERE status='failed'";
        if ($selectedPeriodDays > 0 && $jobHasCreatedAt) {
            $totalSql .= " AND created_at >= DATE_SUB(NOW(), INTERVAL " . (int)$selectedPeriodDays . " DAY)";
        }
        $stTotal = $pdo->query($totalSql);
        $totalFailedJobs = (int)$stTotal->fetchColumn();
        $stTotal->closeCursor();

        if ($jobErrorColumn) {
            $sqlErr = "SELECT
                SUM(CASE WHEN LOWER(COALESCE($jobErrorColumn,'')) LIKE '%ffmpeg%' OR LOWER(COALESCE($jobErrorColumn,'')) LIKE '%ffprobe%' THEN 1 ELSE 0 END) AS ffmpeg_count,
                SUM(CASE WHEN LOWER(COALESCE($jobErrorColumn,'')) LIKE '%storage%' OR LOWER(COALESCE($jobErrorColumn,'')) LIKE '%s3%' OR LOWER(COALESCE($jobErrorColumn,'')) LIKE '%r2%' THEN 1 ELSE 0 END) AS storage_count,
                SUM(CASE WHEN LOWER(COALESCE($jobErrorColumn,'')) LIKE '%curl%' OR LOWER(COALESCE($jobErrorColumn,'')) LIKE '%timeout%' OR LOWER(COALESCE($jobErrorColumn,'')) LIKE '%dns%' THEN 1 ELSE 0 END) AS network_count,
                SUM(CASE WHEN LOWER(COALESCE($jobErrorColumn,'')) LIKE '%sqlstate%' OR LOWER(COALESCE($jobErrorColumn,'')) LIKE '%database%' THEN 1 ELSE 0 END) AS db_count,
                SUM(CASE WHEN LOWER(COALESCE($jobErrorColumn,'')) LIKE '%webhook%' THEN 1 ELSE 0 END) AS webhook_count
            FROM video_jobs
            WHERE status='failed'";
            if ($selectedPeriodDays > 0 && $jobHasCreatedAt) {
                $sqlErr .= " AND created_at >= DATE_SUB(NOW(), INTERVAL " . (int)$selectedPeriodDays . " DAY)";
            }

            $stErr = $pdo->query($sqlErr);
            $rowErr = $stErr->fetch(PDO::FETCH_ASSOC) ?: [];
            $stErr->closeCursor();

            $errorTypeCounts['ffmpeg'] = (int)($rowErr['ffmpeg_count'] ?? 0);
            $errorTypeCounts['storage'] = (int)($rowErr['storage_count'] ?? 0);
            $errorTypeCounts['network'] = (int)($rowErr['network_count'] ?? 0);
            $errorTypeCounts['db'] = (int)($rowErr['db_count'] ?? 0);
            $errorTypeCounts['webhook'] = (int)($rowErr['webhook_count'] ?? 0);

            $known = $errorTypeCounts['ffmpeg'] + $errorTypeCounts['storage'] + $errorTypeCounts['network'] + $errorTypeCounts['db'] + $errorTypeCounts['webhook'];
            $errorTypeCounts['other'] = max(0, $totalFailedJobs - $known);
        } else {
            $errorTypeCounts['other'] = $totalFailedJobs;
        }
    }
} catch (Throwable $e) {
    // mantém contadores em zero em caso de falha
}


    // Série diária de falhas por tipo (gráfico 7/30 dias)
    try {
        if (($checks['table_video_jobs'] ?? false) && $jobHasCreatedAt) {
            $daysWindow = max(7, min(30, (int)$selectedTrendDays));

            $trendMap = [];
            for ($i = $daysWindow - 1; $i >= 0; $i--) {
                $dObj = (new DateTimeImmutable('today'))->modify('-' . $i . ' day');
                $d = $dObj->format('Y-m-d');
                $trendMap[$d] = [
                    'date' => $d,
                    'label' => $dObj->format('d/m'),
                    'ffmpeg' => 0,
                    'storage' => 0,
                    'network' => 0,
                    'db' => 0,
                    'webhook' => 0,
                    'other' => 0,
                    'total' => 0,
                ];
            }

            if ($jobErrorColumn) {
                $sqlTrend = "SELECT DATE(created_at) AS d,
                    SUM(CASE WHEN LOWER(COALESCE($jobErrorColumn,'')) LIKE '%ffmpeg%' OR LOWER(COALESCE($jobErrorColumn,'')) LIKE '%ffprobe%' THEN 1 ELSE 0 END) AS ffmpeg_count,
                    SUM(CASE WHEN LOWER(COALESCE($jobErrorColumn,'')) LIKE '%storage%' OR LOWER(COALESCE($jobErrorColumn,'')) LIKE '%s3%' OR LOWER(COALESCE($jobErrorColumn,'')) LIKE '%r2%' THEN 1 ELSE 0 END) AS storage_count,
                    SUM(CASE WHEN LOWER(COALESCE($jobErrorColumn,'')) LIKE '%curl%' OR LOWER(COALESCE($jobErrorColumn,'')) LIKE '%timeout%' OR LOWER(COALESCE($jobErrorColumn,'')) LIKE '%dns%' THEN 1 ELSE 0 END) AS network_count,
                    SUM(CASE WHEN LOWER(COALESCE($jobErrorColumn,'')) LIKE '%sqlstate%' OR LOWER(COALESCE($jobErrorColumn,'')) LIKE '%database%' THEN 1 ELSE 0 END) AS db_count,
                    SUM(CASE WHEN LOWER(COALESCE($jobErrorColumn,'')) LIKE '%webhook%' THEN 1 ELSE 0 END) AS webhook_count,
                    COUNT(*) AS total_count
                FROM video_jobs
                WHERE status='failed'
                  AND created_at >= DATE_SUB(CURDATE(), INTERVAL {$daysWindow} DAY)
                GROUP BY DATE(created_at)
                ORDER BY DATE(created_at) ASC";

                $stTrend = $pdo->query($sqlTrend);
                $rowsTrend = $stTrend->fetchAll(PDO::FETCH_ASSOC);
                $stTrend->closeCursor();

                foreach ($rowsTrend as $tr) {
                    $d = (string)($tr['d'] ?? '');
                    if (!isset($trendMap[$d])) continue;

                    $ff = (int)($tr['ffmpeg_count'] ?? 0);
                    $st = (int)($tr['storage_count'] ?? 0);
                    $nw = (int)($tr['network_count'] ?? 0);
                    $db = (int)($tr['db_count'] ?? 0);
                    $wh = (int)($tr['webhook_count'] ?? 0);
                    $tt = (int)($tr['total_count'] ?? 0);
                    $known = $ff + $st + $nw + $db + $wh;
                    $ot = max(0, $tt - $known);

                    $trendMap[$d]['ffmpeg'] = $ff;
                    $trendMap[$d]['storage'] = $st;
                    $trendMap[$d]['network'] = $nw;
                    $trendMap[$d]['db'] = $db;
                    $trendMap[$d]['webhook'] = $wh;
                    $trendMap[$d]['other'] = $ot;
                    $trendMap[$d]['total'] = $tt;
                }
            } else {
                $sqlTrend = "SELECT DATE(created_at) AS d, COUNT(*) AS total_count
                             FROM video_jobs
                             WHERE status='failed'
                               AND created_at >= DATE_SUB(CURDATE(), INTERVAL {$daysWindow} DAY)
                             GROUP BY DATE(created_at)
                             ORDER BY DATE(created_at) ASC";
                $stTrend = $pdo->query($sqlTrend);
                $rowsTrend = $stTrend->fetchAll(PDO::FETCH_ASSOC);
                $stTrend->closeCursor();
                foreach ($rowsTrend as $tr) {
                    $d = (string)($tr['d'] ?? '');
                    if (!isset($trendMap[$d])) continue;
                    $tt = (int)($tr['total_count'] ?? 0);
                    $trendMap[$d]['other'] = $tt;
                    $trendMap[$d]['total'] = $tt;
                }
            }

            $errorTrendSeries = array_values($trendMap);
            foreach ($errorTrendSeries as $r) {
                $maxTrendTotal = max($maxTrendTotal, (int)($r['total'] ?? 0));
            }
            if ($maxTrendTotal < 1) $maxTrendTotal = 1;
        }
    } catch (Throwable $e) {
        $errorTrendSeries = [];
        $maxTrendTotal = 1;
    }

    $paymentSettings = getPaymentSettings($pdo);

    $mpToken = envGet($env, 'MERCADOPAGO_ACCESS_TOKEN') ?? ($paymentSettings['mercadopago']['access_token'] ?? null);
    $checks['mp_config'] = !empty($mpToken);
    $details['mp'] = $checks['mp_config'] ? 'Token encontrado.' : 'Token ausente em .env e payment_settings.';

    $pgToken = envGet($env, 'PAGBANK_TOKEN')
        ?? envGet($env, 'PAGSEGURO_TOKEN')
        ?? ($paymentSettings['pagbank']['token'] ?? null)
        ?? ($paymentSettings['pagseguro']['token'] ?? null);
    $checks['pagbank_config'] = !empty($pgToken);
    $details['pagbank'] = $checks['pagbank_config'] ? 'Token encontrado.' : 'Token ausente em .env e payment_settings.';

    $manualEnabled = envGet($env, 'MANUAL_PAYMENT_ENABLED');
    if ($manualEnabled === null) $manualEnabled = $paymentSettings['manual']['enabled'] ?? null;
    $checks['manual_payment_enabled'] = in_array((string)$manualEnabled, ['1','true','on','yes'], true);
    $details['manual_payment'] = $checks['manual_payment_enabled'] ? 'Pagamento manual habilitado.' : 'Pagamento manual desabilitado (enabled=1).';

    try {
        $details['count_users'] = (string)$pdo->query("SELECT COUNT(*) FROM users")->fetchColumn();
        $details['count_plans'] = (string)$pdo->query("SELECT COUNT(*) FROM plans")->fetchColumn();
        $details['count_subscriptions'] = (string)$pdo->query("SELECT COUNT(*) FROM subscriptions")->fetchColumn();
        $details['count_videos'] = (string)$pdo->query("SELECT COUNT(*) FROM videos")->fetchColumn();
        $details['count_jobs_pending'] = (string)$pdo->query("SELECT COUNT(*) FROM video_jobs WHERE status IN ('queued','pending','processing')")->fetchColumn();
        $details['count_jobs_failed'] = (string)$pdo->query("SELECT COUNT(*) FROM video_jobs WHERE status='failed'")->fetchColumn();
    } catch (Throwable $e) {
        // ignore metrics errors
    }
}

// --- FFmpeg / FFprobe ---
$ffmpegBin = envGet($env, 'FFMPEG_BIN');
$ffprobeBin = envGet($env, 'FFPROBE_BIN');
[$ffOk, $ffMsg, $ffResolved] = checkBinary($ffmpegBin, 'where ffmpeg');
[$fpOk, $fpMsg, $fpResolved] = checkBinary($ffprobeBin, 'where ffprobe');

$checks['ffmpeg'] = $ffOk;
$checks['ffprobe'] = $fpOk;
$details['ffmpeg'] = $ffMsg . ($ffResolved ? "<br><small>" . h($ffResolved) . "</small>" : '');
$details['ffprobe'] = $fpMsg . ($fpResolved ? "<br><small>" . h($fpResolved) . "</small>" : '');

// --- Permissões ---
foreach ($paths as $label => $abs) {
    $exists = is_dir($abs);
    $writable = $exists && is_writable($abs);
    $k = 'perm_' . $label;
    $checks[$k] = $exists && $writable;
    $baseMsg = (!$exists ? 'Pasta não existe' : (!$writable ? 'Sem permissão de escrita' : 'OK'));
    if (isset($permFixFeedback[$label])) {
        $baseMsg .= ' | ' . $permFixFeedback[$label]['msg'];
    }
    $details[$k] = $baseMsg . "<br><small>" . h($abs) . "</small>";
}

// --- Score / Semáforo ---
$total = count($checks);
$okCount = count(array_filter($checks, fn($v) => $v === true));
$score = $total > 0 ? (int)round(($okCount / $total) * 100) : 0;

$criticalKeys = ['db_connection','table_users','table_plans','table_subscriptions','table_videos','table_video_jobs','ffmpeg','ffprobe','perm_public/uploads','perm_storage/hls','perm_storage/logs'];
$criticalOk = true;
foreach ($criticalKeys as $ck) {
    if (!($checks[$ck] ?? false)) { $criticalOk = false; break; }
}

if ($criticalOk && $score >= 95) {
    $semaforo = 'green';
} elseif ($criticalOk || $score >= 70) {
    $semaforo = 'yellow';
} else {
    $semaforo = 'red';
}

function semaClass(string $current, string $target): string {
    return $current === $target ? 'on' : 'off';
}

?><!doctype html>
<html lang="pt-br">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Validação do Sistema</title>
  <link rel="stylesheet" href="../assets/vendor/offline-adminlte-lite.css">
  <link rel="stylesheet" href="../assets/vendor/offline-fontawesome-lite.css">
  <style>
    body{background:linear-gradient(135deg,#0f172a,#111827,#1f2937);min-height:100vh;color:#e5e7eb}
    .card{background:rgba(17,24,39,.82);border:1px solid rgba(255,255,255,.12);backdrop-filter: blur(6px)}
    .progress{height:14px}
    .table td,.table th{vertical-align:middle}
    .text-soft{color:#cbd5e1}
    .semaforo-wrap{display:flex;align-items:center;gap:18px;flex-wrap:wrap}
    .semaforo{display:flex;gap:10px;padding:8px 10px;background:rgba(255,255,255,.06);border-radius:999px;border:1px solid rgba(255,255,255,.15)}
    .luz{width:18px;height:18px;border-radius:50%;opacity:.25;box-shadow:0 0 0 1px rgba(255,255,255,.25) inset}
    .luz.verde{background:#22c55e}
    .luz.amarela{background:#f59e0b}
    .luz.vermelha{background:#ef4444}
    .luz.on{opacity:1;box-shadow:0 0 12px currentColor, 0 0 0 1px rgba(255,255,255,.3) inset}
    .badge-pill{border-radius:999px}
    .trend-legend{display:flex;gap:14px;flex-wrap:wrap;margin-bottom:12px}
    .trend-item{display:flex;align-items:center;gap:6px;font-size:.86rem;color:#cbd5e1}
    .trend-dot{width:10px;height:10px;border-radius:50%}
    .trend-grid{display:flex;flex-direction:column;gap:8px}
    .trend-row{display:grid;grid-template-columns:62px 1fr 56px;align-items:center;gap:10px}
    .trend-date{font-size:.82rem;color:#94a3b8}
    .trend-bar{height:14px;border-radius:999px;overflow:hidden;background:rgba(148,163,184,.22);display:flex}
    .trend-seg{height:100%}
    .trend-ffmpeg{background:#dc3545}
    .trend-storage{background:#ffc107}
    .trend-network{background:#17a2b8}
    .trend-db{background:#007bff}
    .trend-webhook{background:#6c757d}
    .trend-other{background:#343a40}
    .trend-total{font-size:.82rem;color:#e2e8f0;text-align:right}
  </style>
</head>
<body>
<div class="container py-4">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h3 class="m-0"><i class="fas fa-shield-alt mr-2"></i>Validação do Sistema</h3>
    <div>
      <a href="?" class="btn btn-sm btn-primary"><i class="fas fa-sync mr-1"></i>Revalidar</a>
      <a href="../dashboard.php" class="btn btn-sm btn-outline-light">Dashboard</a>
    </div>
  </div>

  <div class="card mb-3">
    <div class="card-body">
      <div class="semaforo-wrap mb-2">
        <div class="semaforo" title="Semáforo de saúde geral">
          <span class="luz vermelha <?= semaClass($semaforo,'red') ?>"></span>
          <span class="luz amarela <?= semaClass($semaforo,'yellow') ?>"></span>
          <span class="luz verde <?= semaClass($semaforo,'green') ?>"></span>
        </div>
        <div>
          <?php if ($semaforo === 'green'): ?>
            <span class="badge badge-success badge-pill px-3 py-2">VERDE • Sistema saudável</span>
          <?php elseif ($semaforo === 'yellow'): ?>
            <span class="badge badge-warning badge-pill px-3 py-2">AMARELO • Atenção em itens não críticos</span>
          <?php else: ?>
            <span class="badge badge-danger badge-pill px-3 py-2">VERMELHO • Corrigir itens críticos</span>
          <?php endif; ?>
        </div>
      </div>

      <div class="d-flex justify-content-between mb-2">
        <strong>Saúde geral</strong>
        <strong><?= h((string)$score) ?>%</strong>
      </div>
      <div class="progress mb-2">
        <div class="progress-bar <?= $semaforo === 'green' ? 'bg-success' : ($semaforo === 'yellow' ? 'bg-warning' : 'bg-danger') ?>" role="progressbar" style="width: <?= h((string)$score) ?>%"></div>
      </div>
      <div class="text-soft"><?= $okCount ?> de <?= $total ?> verificações aprovadas.</div>
    </div>
  </div>



<div class="card mb-3">
  <div class="card-header d-flex justify-content-between align-items-center">
    <strong><i class="fas fa-bug mr-1"></i>Falhas por tipo de erro</strong>
    <small class="text-soft">
      Total de jobs falhos: <strong><?= (int)$totalFailedJobs ?></strong>
      <?php if ($selectedPeriodDays > 0): ?> · Período: últimos <?= (int)$selectedPeriodDays ?> dias<?php endif; ?>
    </small>
  </div>
  <div class="card-body">
    <div class="row">
      <div class="col-lg-2 col-md-4 col-6 mb-2">
        <div class="small-box bg-danger">
          <div class="inner">
            <h3><?= (int)($errorTypeCounts['ffmpeg'] ?? 0) ?></h3>
            <p>FFmpeg/FFprobe</p>
          </div>
          <div class="icon"><i class="fas fa-film"></i></div>
        </div>
      </div>
      <div class="col-lg-2 col-md-4 col-6 mb-2">
        <div class="small-box bg-warning">
          <div class="inner">
            <h3><?= (int)($errorTypeCounts['storage'] ?? 0) ?></h3>
            <p>Storage</p>
          </div>
          <div class="icon"><i class="fas fa-database"></i></div>
        </div>
      </div>
      <div class="col-lg-2 col-md-4 col-6 mb-2">
        <div class="small-box bg-info">
          <div class="inner">
            <h3><?= (int)($errorTypeCounts['network'] ?? 0) ?></h3>
            <p>Rede</p>
          </div>
          <div class="icon"><i class="fas fa-wifi"></i></div>
        </div>
      </div>
      <div class="col-lg-2 col-md-4 col-6 mb-2">
        <div class="small-box bg-primary">
          <div class="inner">
            <h3><?= (int)($errorTypeCounts['db'] ?? 0) ?></h3>
            <p>Banco</p>
          </div>
          <div class="icon"><i class="fas fa-server"></i></div>
        </div>
      </div>
      <div class="col-lg-2 col-md-4 col-6 mb-2">
        <div class="small-box bg-secondary">
          <div class="inner">
            <h3><?= (int)($errorTypeCounts['webhook'] ?? 0) ?></h3>
            <p>Webhook</p>
          </div>
          <div class="icon"><i class="fas fa-plug"></i></div>
        </div>
      </div>
      <div class="col-lg-2 col-md-4 col-6 mb-2">
        <div class="small-box bg-dark">
          <div class="inner">
            <h3><?= (int)($errorTypeCounts['other'] ?? 0) ?></h3>
            <p>Outros</p>
          </div>
          <div class="icon"><i class="fas fa-question-circle"></i></div>
        </div>
      </div>
    </div>
    <small class="text-soft">Os contadores são calculados com base no conteúdo de <code>error_message</code> ou <code>last_error</code> dos jobs com status <code>failed</code>.</small>
  </div>
</div>

<div class="card mb-3" id="errorTrendCard">
  <div class="card-header d-flex justify-content-between align-items-center flex-wrap">
    <strong><i class="fas fa-chart-bar mr-1"></i>Gráfico de falhas por tipo (últimos 7/30 dias)</strong>
    <form method="get" class="form-inline m-0">
      <input type="hidden" name="period" value="<?= h($selectedPeriod) ?>">
      <input type="hidden" name="plan_id" value="<?= (int)$selectedPlanId ?>">
      <input type="hidden" name="user_id" value="<?= (int)$selectedUserId ?>">
      <label class="mr-2 mb-0 text-soft">Janela</label>
      <select name="trend_period" class="form-control form-control-sm mr-2">
        <option value="7" <?= ($selectedTrendPeriod === '7' ? 'selected' : '') ?>>7 dias</option>
        <option value="30" <?= ($selectedTrendPeriod === '30' ? 'selected' : '') ?>>30 dias</option>
      </select>
      <button class="btn btn-sm btn-outline-light" type="submit"><i class="fas fa-filter mr-1"></i>Atualizar</button>
    </form>
    <div class="btn-group btn-group-sm ml-2 mt-2 mt-md-0" role="group" aria-label="Exportar gráfico">
      <button type="button" class="btn btn-outline-success" id="btnExportTrendPng"><i class="fas fa-file-image mr-1"></i>Exportar PNG</button>
      <button type="button" class="btn btn-outline-danger" id="btnExportTrendPdf"><i class="fas fa-file-pdf mr-1"></i>Exportar PDF</button>
    </div>
  </div>
  <div class="card-body">
    <?php if (!($checks['table_video_jobs'] ?? false)): ?>
      <div class="alert alert-warning mb-0">Tabela <code>video_jobs</code> não encontrada para gerar gráfico.</div>
    <?php elseif (!$jobHasCreatedAt): ?>
      <div class="alert alert-warning mb-0">A coluna <code>video_jobs.created_at</code> não existe. O gráfico diário requer essa coluna.</div>
    <?php else: ?>
      <div class="trend-legend">
        <span class="trend-item"><span class="trend-dot trend-ffmpeg"></span>FFmpeg</span>
        <span class="trend-item"><span class="trend-dot trend-storage"></span>Storage</span>
        <span class="trend-item"><span class="trend-dot trend-network"></span>Rede</span>
        <span class="trend-item"><span class="trend-dot trend-db"></span>Banco</span>
        <span class="trend-item"><span class="trend-dot trend-webhook"></span>Webhook</span>
        <span class="trend-item"><span class="trend-dot trend-other"></span>Outros</span>
      </div>

      <div class="trend-grid">
        <?php foreach (($errorTrendSeries ?? []) as $tr): ?>
          <?php
            $tt = (int)($tr['total'] ?? 0);
            $base = max(1, (int)$maxTrendTotal);
            $wf = ((int)($tr['ffmpeg'] ?? 0) / $base) * 100;
            $ws = ((int)($tr['storage'] ?? 0) / $base) * 100;
            $wn = ((int)($tr['network'] ?? 0) / $base) * 100;
            $wd = ((int)($tr['db'] ?? 0) / $base) * 100;
            $ww = ((int)($tr['webhook'] ?? 0) / $base) * 100;
            $wo = ((int)($tr['other'] ?? 0) / $base) * 100;
          ?>
          <div class="trend-row" title="<?= h((string)($tr['date'] ?? '')) ?> | total: <?= $tt ?>">
            <div class="trend-date"><?= h((string)($tr['label'] ?? '')) ?></div>
            <div class="trend-bar">
              <?php if ($wf > 0): ?><span class="trend-seg trend-ffmpeg" style="width:<?= number_format($wf, 4, '.', '') ?>%"></span><?php endif; ?>
              <?php if ($ws > 0): ?><span class="trend-seg trend-storage" style="width:<?= number_format($ws, 4, '.', '') ?>%"></span><?php endif; ?>
              <?php if ($wn > 0): ?><span class="trend-seg trend-network" style="width:<?= number_format($wn, 4, '.', '') ?>%"></span><?php endif; ?>
              <?php if ($wd > 0): ?><span class="trend-seg trend-db" style="width:<?= number_format($wd, 4, '.', '') ?>%"></span><?php endif; ?>
              <?php if ($ww > 0): ?><span class="trend-seg trend-webhook" style="width:<?= number_format($ww, 4, '.', '') ?>%"></span><?php endif; ?>
              <?php if ($wo > 0): ?><span class="trend-seg trend-other" style="width:<?= number_format($wo, 4, '.', '') ?>%"></span><?php endif; ?>
            </div>
            <div class="trend-total"><?= $tt ?></div>
          </div>
        <?php endforeach; ?>
      </div>

      <small class="text-soft d-block mt-2">
        Escala: a barra completa representa o dia com maior número de falhas da janela selecionada (<?= (int)$selectedTrendDays ?> dias).
      </small>
    <?php endif; ?>
  </div>
</div>

  <div class="card mb-3">
    <div class="card-header d-flex justify-content-between align-items-center">
      <strong><i class="fas fa-screwdriver-wrench mr-1"></i>Ações de correção</strong>
      <div class="d-flex align-items-center">
        <form method="post" class="m-0">
          <input type="hidden" name="action" value="fix_permissions">
          <button type="submit" class="btn btn-sm btn-warning"><i class="fas fa-wand-magic-sparkles mr-1"></i>Corrigir permissões automaticamente</button>
        </form>
        <form method="post" class="m-0 ml-2">
          <input type="hidden" name="action" value="reprocess_failed_jobs">
          <button type="submit" class="btn btn-sm btn-danger"><i class="fas fa-rotate-left mr-1"></i>Reprocessar jobs com falha</button>
        </form>
        <form method="post" class="m-0 ml-2">
          <input type="hidden" name="action" value="reprocess_failed_jobs_ffmpeg_only">
          <input type="hidden" name="period" value="<?= h($selectedPeriod) ?>">
          <button type="submit" class="btn btn-sm btn-outline-light"><i class="fas fa-film mr-1"></i>Reprocessar somente FFmpeg</button>
        </form>
        <a href="retry_attempts.php" class="btn btn-sm btn-outline-info ml-2"><i class="fas fa-wave-square mr-1"></i>Histórico FFmpeg</a>
      </div>
    </div>
    <div class="card-body border-top">
      <form method="post" class="form-inline flex-wrap align-items-end">

        <div class="mr-2 mb-2">
          <label class="mb-1 d-block">Plano</label>
          <select name="plan_id" class="form-control form-control-sm" style="min-width:220px">
            <option value="0">Todos os planos</option>
            <?php foreach (($plansForFilter ?? []) as $pl): ?>
              <?php $pid = (int)($pl['id'] ?? 0); ?>
              <option value="<?= $pid ?>" <?= ($selectedPlanId === $pid ? 'selected' : '') ?>>
                #<?= $pid ?> - <?= h((string)($pl['name'] ?? 'Plano')) ?>
              </option>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="mr-2 mb-2">
          <label class="mb-1 d-block">Período</label>
          <select name="period" class="form-control form-control-sm" style="min-width:180px">
            <option value="all" <?= ($selectedPeriod === 'all' ? 'selected' : '') ?>>Todo histórico</option>
            <option value="7" <?= ($selectedPeriod === '7' ? 'selected' : '') ?>>Últimos 7 dias</option>
            <option value="30" <?= ($selectedPeriod === '30' ? 'selected' : '') ?>>Últimos 30 dias</option>
          </select>
        </div>

        <div class="mr-2 mb-2">
          <label class="mb-1 d-block">Usuário</label>
          <select name="user_id" class="form-control form-control-sm" style="min-width:360px" required>
            <option value="">Selecione um usuário com jobs falhos</option>
            <?php foreach (($usersWithFailedJobs ?? []) as $uf): ?>
              <?php $uid = (int)($uf['id'] ?? 0); ?>
              <?php $planLabel = trim((string)($uf['plan_name'] ?? 'Sem plano')); ?>
              <option value="<?= $uid ?>" <?= ($selectedUserId === $uid ? 'selected' : '') ?>>
                #<?= $uid ?> - <?= h((string)($uf['name'] ?? '')) ?> (<?= h((string)($uf['email'] ?? '')) ?>) - plano: <?= h($planLabel) ?> - falhas: <?= h((string)($uf['failed_count'] ?? '0')) ?>
              </option>
            <?php endforeach; ?>
          </select>
        </div>

        <button type="submit" name="action" value="reprocess_failed_jobs_user" class="btn btn-sm btn-outline-danger mb-2 mr-2">
          <i class="fas fa-user-gear mr-1"></i>Reprocessar falhas do usuário
        </button>

        <button type="submit" name="action" value="reprocess_failed_jobs_plan_user" class="btn btn-sm btn-outline-primary mb-2">
          <i class="fas fa-user-check mr-1"></i>Reprocessar plano + usuário
        </button>

</form>

<form method="post" class="form-inline flex-wrap align-items-end mt-2">
  <input type="hidden" name="action" value="reprocess_failed_jobs_plan">

  <div class="mr-2 mb-2">
    <label class="mb-1 d-block">Plano (lote)</label>
    <select name="plan_id" class="form-control form-control-sm" style="min-width:220px" required>
      <option value="">Selecione um plano</option>
      <?php foreach (($plansForFilter ?? []) as $pl): ?>
        <?php $pid = (int)($pl['id'] ?? 0); ?>
        <option value="<?= $pid ?>" <?= ($selectedPlanId === $pid ? 'selected' : '') ?>>
          #<?= $pid ?> - <?= h((string)($pl['name'] ?? 'Plano')) ?>
        </option>
      <?php endforeach; ?>
    </select>
  </div>

  <div class="mr-2 mb-2">
    <label class="mb-1 d-block">Período</label>
    <select name="period" class="form-control form-control-sm" style="min-width:180px">
      <option value="all" <?= ($selectedPeriod === 'all' ? 'selected' : '') ?>>Todo histórico</option>
      <option value="7" <?= ($selectedPeriod === '7' ? 'selected' : '') ?>>Últimos 7 dias</option>
      <option value="30" <?= ($selectedPeriod === '30' ? 'selected' : '') ?>>Últimos 30 dias</option>
    </select>
  </div>

  <button type="submit" class="btn btn-sm btn-outline-warning mb-2">
    <i class="fas fa-layer-group mr-1"></i>Reprocessar por plano
  </button>
</form>
<small class="text-soft">
        Use os botões para reprocessar por usuário, por plano + usuário, ou por plano (lote). O filtro por plano usa <code>users.current_plan_id</code>. O filtro por período usa a data de criação do job
        <?php if (!$jobHasCreatedAt): ?><span class="text-warning">(coluna <code>video_jobs.created_at</code> não encontrada, período ignorado).</span><?php endif; ?>
      </small>
    </div>
    <div class="card-body">
      <p class="mb-2 text-soft">
        Este botão tenta criar as pastas críticas e validar escrita em:
        <code>public/uploads</code>, <code>storage/hls</code> e <code>storage/logs</code>.
      </p>
      <?php if (!empty($permFixFeedback)): ?>
        <div class="alert alert-info mb-2">
          <strong>Resultado da correção de permissões:</strong>
          <ul class="mb-0 mt-2">
            <?php foreach ($permFixFeedback as $label => $r): ?>
              <li>
                <?= h($label) ?>:
                <?= $r['ok'] ? "<span class='text-success'>OK</span>" : "<span class='text-danger'>Falhou</span>" ?>
                <small class="d-block"><?= h($r['msg']) ?></small>
              </li>
            <?php endforeach; ?>
          </ul>
        </div>
      <?php endif; ?>

      <?php if (is_array($jobFixFeedback)): ?>
        <div class="alert <?= $jobFixFeedback['ok'] ? 'alert-success' : 'alert-danger' ?> mb-0">
          <strong><?= $jobFixFeedback['ok'] ? 'Reprocessamento concluído' : 'Falha no reprocessamento' ?>:</strong>
          <span><?= h((string)$jobFixFeedback['msg']) ?></span>
        </div>
      <?php endif; ?>

      <?php if (is_array($jobFixFfmpegFeedback)): ?>
        <div class="alert <?= $jobFixFfmpegFeedback['ok'] ? 'alert-success' : 'alert-danger' ?> mt-2 mb-0">
          <strong><?= $jobFixFfmpegFeedback['ok'] ? 'Reprocessamento FFmpeg concluído' : 'Falha no reprocessamento FFmpeg' ?>:</strong>
          <span><?= h((string)$jobFixFfmpegFeedback['msg']) ?></span>
        </div>
      <?php endif; ?>


<?php if (is_array($jobFixByUserFeedback)): ?>
  <div class="alert <?= $jobFixByUserFeedback['ok'] ? 'alert-success' : 'alert-danger' ?> mt-2 mb-0">
    <strong><?= $jobFixByUserFeedback['ok'] ? 'Reprocessamento por usuário concluído' : 'Falha no reprocessamento por usuário' ?>:</strong>
    <span><?= h((string)$jobFixByUserFeedback['msg']) ?></span>
  </div>
<?php endif; ?>


<?php if (is_array($jobFixByPlanUserFeedback)): ?>
  <div class="alert <?= $jobFixByPlanUserFeedback['ok'] ? 'alert-success' : 'alert-danger' ?> mt-2 mb-0">
    <strong><?= $jobFixByPlanUserFeedback['ok'] ? 'Reprocessamento por plano + usuário concluído' : 'Falha no reprocessamento por plano + usuário' ?>:</strong>
    <span><?= h((string)$jobFixByPlanUserFeedback['msg']) ?></span>
  </div>
<?php endif; ?>


<?php if (is_array($jobFixByPlanFeedback)): ?>
  <div class="alert <?= $jobFixByPlanFeedback['ok'] ? 'alert-success' : 'alert-danger' ?> mt-2 mb-0">
    <strong><?= $jobFixByPlanFeedback['ok'] ? 'Reprocessamento por plano concluído' : 'Falha no reprocessamento por plano' ?>:</strong>
    <span><?= h((string)$jobFixByPlanFeedback['msg']) ?></span>
  </div>
<?php endif; ?>
    </div>
  </div>

  <div class="row">
    <div class="col-lg-6">
      <div class="card mb-3">
        <div class="card-header"><strong><i class="fas fa-database mr-1"></i>Banco de Dados</strong></div>
        <div class="card-body p-0">
          <table class="table table-dark table-striped mb-0">
            <tbody>
              <tr>
                <td>Conexão MySQL</td>
                <td><?= statusBadge($checks['db_connection'] ?? false) ?></td>
              </tr>
              <?php if (!($checks['db_connection'] ?? false)): ?>
              <tr><td colspan="2" class="text-danger"><small><?= h((string)$dbError) ?></small></td></tr>
              <?php endif; ?>
              <?php foreach ($requiredTables as $tb): $k = 'table_' . $tb; ?>
              <tr>
                <td>Tabela <code><?= h($tb) ?></code></td>
                <td><?= statusBadge($checks[$k] ?? false) ?></td>
              </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
      </div>

      <div class="card mb-3">
        <div class="card-header"><strong><i class="fas fa-credit-card mr-1"></i>Pagamentos</strong></div>
        <div class="card-body p-0">
          <table class="table table-dark table-striped mb-0">
            <tbody>
              <tr><td>Mercado Pago configurado</td><td><?= statusBadge($checks['mp_config'] ?? false) ?></td></tr>
              <tr><td colspan="2"><small><?= $details['mp'] ?? '-' ?></small></td></tr>
              <tr><td>PagBank/PagSeguro configurado</td><td><?= statusBadge($checks['pagbank_config'] ?? false) ?></td></tr>
              <tr><td colspan="2"><small><?= $details['pagbank'] ?? '-' ?></small></td></tr>
              <tr><td>Pagamento manual habilitado</td><td><?= statusBadge($checks['manual_payment_enabled'] ?? false) ?></td></tr>
              <tr><td colspan="2"><small><?= $details['manual_payment'] ?? '-' ?></small></td></tr>
            </tbody>
          </table>
        </div>
      </div>
    </div>

    <div class="col-lg-6">
      <div class="card mb-3">
        <div class="card-header"><strong><i class="fas fa-film mr-1"></i>FFmpeg / FFprobe</strong></div>
        <div class="card-body p-0">
          <table class="table table-dark table-striped mb-0">
            <tbody>
              <tr><td>FFmpeg</td><td><?= statusBadge($checks['ffmpeg'] ?? false) ?></td></tr>
              <tr><td colspan="2"><small><?= $details['ffmpeg'] ?? '-' ?></small></td></tr>
              <tr><td>FFprobe</td><td><?= statusBadge($checks['ffprobe'] ?? false) ?></td></tr>
              <tr><td colspan="2"><small><?= $details['ffprobe'] ?? '-' ?></small></td></tr>
            </tbody>
          </table>
        </div>
      </div>

      <div class="card mb-3">
        <div class="card-header"><strong><i class="fas fa-folder-open mr-1"></i>Permissões de Pasta</strong></div>
        <div class="card-body p-0">
          <table class="table table-dark table-striped mb-0">
            <tbody>
              <?php foreach ($paths as $label => $_): $k = 'perm_' . $label; ?>
              <tr>
                <td><?= h($label) ?></td>
                <td><?= statusBadge($checks[$k] ?? false) ?></td>
              </tr>
              <tr><td colspan="2"><small><?= $details[$k] ?? '-' ?></small></td></tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
      </div>
    </div>
  </div>

  <div class="card">
    <div class="card-header"><strong><i class="fas fa-chart-line mr-1"></i>Métricas rápidas</strong></div>
    <div class="card-body">
      <div class="row">
        <div class="col-sm-2 col-6 mb-2"><div class="small text-soft">Usuários</div><div class="h5 m-0"><?= h($details['count_users'] ?? '-') ?></div></div>
        <div class="col-sm-2 col-6 mb-2"><div class="small text-soft">Planos</div><div class="h5 m-0"><?= h($details['count_plans'] ?? '-') ?></div></div>
        <div class="col-sm-2 col-6 mb-2"><div class="small text-soft">Assinaturas</div><div class="h5 m-0"><?= h($details['count_subscriptions'] ?? '-') ?></div></div>
        <div class="col-sm-2 col-6 mb-2"><div class="small text-soft">Vídeos</div><div class="h5 m-0"><?= h($details['count_videos'] ?? '-') ?></div></div>
        <div class="col-sm-2 col-6 mb-2"><div class="small text-soft">Jobs pendentes</div><div class="h5 m-0"><?= h($details['count_jobs_pending'] ?? '-') ?></div></div>
        <div class="col-sm-2 col-6 mb-2"><div class="small text-soft">Jobs com falha</div><div class="h5 m-0"><?= h($details['count_jobs_failed'] ?? '-') ?></div></div>
      </div>
    </div>
  </div>

  <p class="mt-3 text-soft">
    Dica: se FFmpeg/FFprobe falhar, configure no <code>.env</code>:
    <code>FFMPEG_BIN=C:\ffmpeg\bin\ffmpeg.exe</code> e <code>FFPROBE_BIN=C:\ffmpeg\bin\ffprobe.exe>.
  </p>
</div>

<script src="../assets/vendor/html2canvas.min.js"></script>
<script src="../assets/vendor/jspdf.umd.min.js"></script>
<script>
(function(){
  const btnPng = document.getElementById('btnExportTrendPng');
  const btnPdf = document.getElementById('btnExportTrendPdf');
  const card = document.getElementById('errorTrendCard');
  if (!btnPng || !btnPdf || !card) return;

  function stamp(){
    const d = new Date();
    const pad = (n)=> String(n).padStart(2,'0');
    return `${d.getFullYear()}${pad(d.getMonth()+1)}${pad(d.getDate())}_${pad(d.getHours())}${pad(d.getMinutes())}`;
  }

  async function renderCard(){
    if (typeof html2canvas === 'undefined') {
      throw new Error('Biblioteca html2canvas não carregada.');
    }
    return await html2canvas(card, {
      scale: 2,
      useCORS: true,
      backgroundColor: '#0f172a'
    });
  }

  btnPng.addEventListener('click', async function(){
    try{
      btnPng.disabled = true;
      const old = btnPng.innerHTML;
      btnPng.innerHTML = '<i class="fas fa-spinner fa-spin mr-1"></i>Gerando PNG...';
      const canvas = await renderCard();
      const a = document.createElement('a');
      a.href = canvas.toDataURL('image/png');
      a.download = `erro_trend_${stamp()}.png`;
      document.body.appendChild(a);
      a.click();
      a.remove();
      btnPng.innerHTML = old;
      btnPng.disabled = false;
    } catch(err){
      alert('Falha ao exportar PNG: ' + (err?.message || err));
      btnPng.disabled = false;
      btnPng.innerHTML = '<i class="fas fa-file-image mr-1"></i>Exportar PNG';
    }
  });

  btnPdf.addEventListener('click', async function(){
    try{
      btnPdf.disabled = true;
      const old = btnPdf.innerHTML;
      btnPdf.innerHTML = '<i class="fas fa-spinner fa-spin mr-1"></i>Gerando PDF...';

      const canvas = await renderCard();
      const imgData = canvas.toDataURL('image/png');

      const jsPDFNS = window.jspdf || {};
      const jsPDFCtor = jsPDFNS.jsPDF;

      if (jsPDFCtor) {
        const pdf = new jsPDFCtor('p', 'mm', 'a4');
        const pageW = pdf.internal.pageSize.getWidth();
        const pageH = pdf.internal.pageSize.getHeight();
        const margin = 8;
        const imgW = pageW - (margin * 2);
        const imgH = (canvas.height * imgW) / canvas.width;

        let heightLeft = imgH;
        let position = margin;
        pdf.addImage(imgData, 'PNG', margin, position, imgW, imgH);
        heightLeft -= (pageH - margin * 2);

        while (heightLeft > 0) {
          pdf.addPage();
          position = margin - (imgH - heightLeft);
          pdf.addImage(imgData, 'PNG', margin, position, imgW, imgH);
          heightLeft -= (pageH - margin * 2);
        }

        pdf.save(`erro_trend_${stamp()}.pdf`);
      } else {
        // Fallback offline sem jsPDF: abre impressão do navegador (Salvar como PDF)
        const w = window.open('', '_blank');
        if (!w) throw new Error('Pop-up bloqueado pelo navegador.');
        w.document.write('<!doctype html><html><head><meta charset="utf-8"><title>Exportar PDF</title>');
        w.document.write('<style>html,body{margin:0;padding:0;background:#fff}img{width:100%;height:auto;display:block}</style>');
        w.document.write('</head><body><img src="' + imgData + '" alt="gráfico" /></body></html>');
        w.document.close();
        w.focus();
        setTimeout(function(){ w.print(); }, 250);
      }
      btnPdf.innerHTML = old;
      btnPdf.disabled = false;
    } catch(err){
      alert('Falha ao exportar PDF: ' + (err?.message || err));
      btnPdf.disabled = false;
      btnPdf.innerHTML = '<i class="fas fa-file-pdf mr-1"></i>Exportar PDF';
    }
  });
})();
</script>

</body>
</html>
