<?php
declare(strict_types=1);
require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../app/helpers.php';
require_admin();

function calc_period_end(string $intervalUnit, int $intervalCount): string {
    $intervalCount = max(1, $intervalCount);
    $start = new DateTimeImmutable('now');
    switch ($intervalUnit) {
        case 'day': $end = $start->modify('+' . $intervalCount . ' day'); break;
        case 'week': $end = $start->modify('+' . $intervalCount . ' week'); break;
        case 'year': $end = $start->modify('+' . $intervalCount . ' year'); break;
        case 'month':
        default: $end = $start->modify('+' . $intervalCount . ' month'); break;
    }
    return $end->format('Y-m-d H:i:s');
}

if (is_post()) {
    if (!csrf_check($_POST['_csrf'] ?? null)) {
        flash('error', 'CSRF inválido.');
        redirect('/admin/users.php');
    }

    $action = (string)($_POST['action'] ?? '');
    if ($action === 'assign_plan') {
        $userId = (int)($_POST['user_id'] ?? 0);
        $planId = (int)($_POST['plan_id'] ?? 0);

        $ust = db()->prepare("SELECT id, role, name FROM users WHERE id=:id LIMIT 1");
        $ust->execute(['id' => $userId]);
        $client = $ust->fetch();

        if (!$client || ($client['role'] ?? '') !== 'client') {
            flash('error', 'Cliente inválido para atribuição de plano.');
            redirect('/admin/users.php');
        }

        if ($planId <= 0) {
            db()->prepare("UPDATE users SET current_plan_id=NULL WHERE id=:id AND role='client'")->execute(['id' => $userId]);
            flash('success', 'Plano removido do cliente com sucesso.');
            redirect('/admin/users.php');
        }

        $pst = db()->prepare("SELECT * FROM plans WHERE id=:id LIMIT 1");
        $pst->execute(['id' => $planId]);
        $plan = $pst->fetch();

        if (!$plan) {
            flash('error', 'Plano não encontrado.');
            redirect('/admin/users.php');
        }

        db()->prepare("UPDATE users SET current_plan_id=:pid, status='active' WHERE id=:uid AND role='client'")
          ->execute(['pid' => $planId, 'uid' => $userId]);

        $periodEnd = calc_period_end((string)$plan['interval_unit'], (int)$plan['interval_count']);
        $adminId = (int)(auth_user()['id'] ?? 0);
        $externalRef = 'admin_assign_' . $userId . '_' . $planId . '_' . time();
        $raw = json_encode(['assigned_by_admin_id' => $adminId, 'source' => 'admin_users_page'], JSON_UNESCAPED_UNICODE);

        try {
            db()->prepare("INSERT INTO subscriptions
                (user_id, plan_id, provider, status, provider_plan_id, external_reference, started_at, current_period_start, current_period_end, raw_payload)
                VALUES
                (:u,:p,'manual','authorized',NULL,:ref,NOW(),NOW(),:end,:raw)")
                ->execute([
                    'u' => $userId,
                    'p' => $planId,
                    'ref' => $externalRef,
                    'end' => $periodEnd,
                    'raw' => $raw
                ]);
        } catch (Throwable $e) {
            // Se o provider manual ainda não existir no ENUM de subscriptions
            flash('warning', 'Plano atribuído, mas não foi possível gravar histórico de assinatura manual. Rode a migration 003.');
            redirect('/admin/users.php');
        }

        flash('success', 'Plano atribuído ao cliente e assinatura manual autorizada com sucesso.');
        redirect('/admin/users.php');
    }
}

if (isset($_GET['toggle'])) {
    $id = (int)$_GET['toggle'];
    if ($id > 0) {
      db()->prepare("UPDATE users SET status = CASE WHEN status='active' THEN 'inactive' ELSE 'active' END WHERE id=:id AND role='client'")
        ->execute(['id' => $id]);
      flash('success', 'Status do cliente atualizado.');
    }
    redirect('/admin/users.php');
}

$plans = db()->query("SELECT id, name, is_enabled, price, currency, interval_unit, interval_count FROM plans ORDER BY is_enabled DESC, price ASC")->fetchAll();

$users = db()->query("
  SELECT u.*, p.name AS plan_name
  FROM users u
  LEFT JOIN plans p ON p.id = u.current_plan_id
  ORDER BY u.id DESC
")->fetchAll();

$pageTitle = 'Admin - Clientes';
include __DIR__ . '/../partials/head.php';
include __DIR__ . '/../partials/navbar.php';
include __DIR__ . '/../partials/sidebar.php';
?>
<div class="content-wrapper">
  <div class="content-header"><div class="container-fluid"><h1>Gerenciar clientes</h1></div></div>
  <section class="content"><div class="container-fluid">
    <?php include __DIR__ . '/../partials/flash.php'; ?>

    <div class="alert alert-info">
      Agora você pode <strong>atribuir um plano manualmente</strong> para qualquer cliente.
      Isso atualiza o plano atual e registra uma assinatura manual autorizada.
    </div>

    <div class="card glass-card">
      <div class="card-body table-responsive">
        <table class="table table-dark table-hover align-middle">
          <thead>
            <tr>
              <th>ID</th>
              <th>Nome</th>
              <th>Email</th>
              <th>Role</th>
              <th>Status</th>
              <th>Plano atual</th>
              <th style="min-width: 320px;">Atribuir plano</th>
              <th>Ações</th>
            </tr>
          </thead>
          <tbody>
          <?php foreach ($users as $u): ?>
            <tr>
              <td><?= (int)$u['id'] ?></td>
              <td><?= e($u['name']) ?></td>
              <td><?= e($u['email']) ?></td>
              <td><?= e($u['role']) ?></td>
              <td><?= e($u['status']) ?></td>
              <td><?= e($u['plan_name'] ?: '—') ?></td>

              <td>
                <?php if (($u['role'] ?? '') === 'client'): ?>
                  <form method="post" class="d-flex gap-2">
                    <input type="hidden" name="_csrf" value="<?= e(csrf_token()) ?>">
                    <input type="hidden" name="action" value="assign_plan">
                    <input type="hidden" name="user_id" value="<?= (int)$u['id'] ?>">
                    <select class="form-select form-select-sm bg-transparent text-light" name="plan_id">
                      <option value="0">Sem plano</option>
                      <?php foreach ($plans as $p): ?>
                        <?php
                          $isCurrent = (int)($u['current_plan_id'] ?? 0) === (int)$p['id'];
                          $label = $p['name'] . ' - R$ ' . number_format((float)$p['price'],2,',','.') . ' / ' . human_interval((string)$p['interval_unit'], (int)$p['interval_count']);
                          if ((int)$p['is_enabled'] !== 1) $label .= ' (OFF)';
                        ?>
                        <option value="<?= (int)$p['id'] ?>" <?= $isCurrent ? 'selected' : '' ?>><?= e($label) ?></option>
                      <?php endforeach; ?>
                    </select>
                    <button class="btn btn-sm btn-outline-info">Salvar</button>
                  </form>
                <?php else: ?>
                  <span class="text-secondary">—</span>
                <?php endif; ?>
              </td>

              <td>
                <?php if (($u['role'] ?? '') === 'client'): ?>
                  <a class="btn btn-sm btn-outline-warning" href="<?= e(url('/admin/users.php?toggle=' . (int)$u['id'])) ?>">Habilitar/Desabilitar</a>
                <?php else: ?>
                  <span class="text-secondary">—</span>
                <?php endif; ?>
              </td>
            </tr>
          <?php endforeach; ?>
          </tbody>
        </table>
      </div>
    </div>
  </div></section>
</div>
<?php include __DIR__ . '/../partials/footer.php'; ?>
