<?php
declare(strict_types=1);

/**
 * fix_admin.php
 * Patch utilitário para criar/atualizar usuário admin no banco.
 * Remova este arquivo após o uso.
 */

function h(?string $s): string {
    return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8');
}

function loadEnv(string $path): array {
    $vars = [];
    if (!is_file($path)) return $vars;
    $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    if (!$lines) return $vars;
    foreach ($lines as $line) {
        $line = trim($line);
        if ($line === '' || $line[0] === '#') continue;
        if (!str_contains($line, '=')) continue;
        [$k, $v] = explode('=', $line, 2);
        $k = trim($k);
        $v = trim($v);
        if ((str_starts_with($v, '"') && str_ends_with($v, '"')) || (str_starts_with($v, "'") && str_ends_with($v, "'"))) {
            $v = substr($v, 1, -1);
        }
        $vars[$k] = $v;
    }
    return $vars;
}

function pickCol(array $cols, array $options): ?string {
    foreach ($options as $opt) {
        if (in_array($opt, $cols, true)) return $opt;
    }
    return null;
}

function nowExpr(): string {
    return 'NOW()';
}

$projectRoot = dirname(__DIR__);
$env = loadEnv($projectRoot . DIRECTORY_SEPARATOR . '.env');

$dbHost = $env['DB_HOST'] ?? '127.0.0.1';
$dbPort = $env['DB_PORT'] ?? '3306';
$dbName = $env['DB_DATABASE'] ?? 'video_saas';
$dbUser = $env['DB_USERNAME'] ?? 'root';
$dbPass = $env['DB_PASSWORD'] ?? '';

$messages = [];
$errors = [];
$done = false;

try {
    $pdo = new PDO(
        "mysql:host={$dbHost};port={$dbPort};dbname={$dbName};charset=utf8mb4",
        $dbUser,
        $dbPass,
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
        ]
    );

    $tableCheck = $pdo->query("SHOW TABLES LIKE 'users'");
    $tableExists = (bool)$tableCheck->fetchColumn();
    $tableCheck->closeCursor();
    if (!$tableExists) {
        throw new RuntimeException("Tabela 'users' não encontrada no banco '{$dbName}'.");
    }

    $colStmt = $pdo->query("SHOW COLUMNS FROM users");
    $cols = [];
    $colMeta = [];
    while ($row = $colStmt->fetch()) {
        $col = (string)$row['Field'];
        $cols[] = $col;
        $colMeta[$col] = $row;
    }
    $colStmt->closeCursor();

    $emailCol = pickCol($cols, ['email', 'mail', 'user_email']);
    $nameCol  = pickCol($cols, ['name', 'nome', 'full_name', 'username']);
    $passCol  = pickCol($cols, ['password_hash', 'password', 'senha', 'passwd']);
    $roleCol  = pickCol($cols, ['role', 'perfil', 'user_type', 'type']);
    $statusCol= pickCol($cols, ['status', 'is_active', 'active', 'enabled']);
    $createdCol = pickCol($cols, ['created_at', 'createdon', 'created']);
    $updatedCol = pickCol($cols, ['updated_at', 'updatedon', 'modified_at', 'updated']);

    if (!$emailCol || !$passCol) {
        throw new RuntimeException("Não foi possível detectar colunas essenciais (email/senha) na tabela users.");
    }

    $defaultEmail = $_POST['email'] ?? 'admin@local.test';
    $defaultName = $_POST['name'] ?? 'Administrador';
    $defaultPassword = $_POST['password'] ?? '123456';

    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $email = trim((string)($_POST['email'] ?? ''));
        $name = trim((string)($_POST['name'] ?? 'Administrador'));
        $password = (string)($_POST['password'] ?? '');
        $deleteAfter = isset($_POST['delete_after']);

        if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errors[] = "Informe um email válido.";
        }
        if (strlen($password) < 6) {
            $errors[] = "A senha deve ter pelo menos 6 caracteres.";
        }

        if (!$errors) {
            $hash = password_hash($password, PASSWORD_BCRYPT);

            $sel = $pdo->prepare("SELECT * FROM users WHERE `{$emailCol}` = :email LIMIT 1");
            $sel->execute([':email' => $email]);
            $existing = $sel->fetch();
            $sel->closeCursor();

            if ($existing) {
                $setParts = [];
                $params = [':email' => $email, ':pass' => $hash];

                $setParts[] = "`{$passCol}` = :pass";
                if ($nameCol) {
                    $setParts[] = "`{$nameCol}` = :name";
                    $params[':name'] = $name;
                }
                if ($roleCol) {
                    $setParts[] = "`{$roleCol}` = :role";
                    $params[':role'] = 'admin';
                }
                if ($statusCol) {
                    $statusType = strtolower((string)($colMeta[$statusCol]['Type'] ?? ''));
                    if (str_contains($statusType, 'tinyint') || str_contains($statusType, 'int') || str_contains($statusType, 'bit') || str_contains($statusType, 'boolean')) {
                        $setParts[] = "`{$statusCol}` = :statusNum";
                        $params[':statusNum'] = 1;
                    } else {
                        $setParts[] = "`{$statusCol}` = :statusTxt";
                        $params[':statusTxt'] = 'active';
                    }
                }
                if ($updatedCol) {
                    $setParts[] = "`{$updatedCol}` = " . nowExpr();
                }

                $sql = "UPDATE users SET " . implode(', ', $setParts) . " WHERE `{$emailCol}` = :email";
                $upd = $pdo->prepare($sql);
                $upd->execute($params);
                $messages[] = "Usuário atualizado com sucesso: {$email}";
            } else {
                $fields = [];
                $values = [];
                $params = [];

                if ($nameCol) {
                    $fields[] = "`{$nameCol}`";
                    $values[] = ":name";
                    $params[':name'] = $name;
                }
                $fields[] = "`{$emailCol}`";
                $values[] = ":email";
                $params[':email'] = $email;

                $fields[] = "`{$passCol}`";
                $values[] = ":pass";
                $params[':pass'] = $hash;

                if ($roleCol) {
                    $fields[] = "`{$roleCol}`";
                    $values[] = ":role";
                    $params[':role'] = 'admin';
                }
                if ($statusCol) {
                    $statusType = strtolower((string)($colMeta[$statusCol]['Type'] ?? ''));
                    $fields[] = "`{$statusCol}`";
                    if (str_contains($statusType, 'tinyint') || str_contains($statusType, 'int') || str_contains($statusType, 'bit') || str_contains($statusType, 'boolean')) {
                        $values[] = ":statusNum";
                        $params[':statusNum'] = 1;
                    } else {
                        $values[] = ":statusTxt";
                        $params[':statusTxt'] = 'active';
                    }
                }
                if ($createdCol) {
                    $fields[] = "`{$createdCol}`";
                    $values[] = nowExpr();
                }
                if ($updatedCol) {
                    $fields[] = "`{$updatedCol}`";
                    $values[] = nowExpr();
                }

                $ins = $pdo->prepare("INSERT INTO users (" . implode(',', $fields) . ") VALUES (" . implode(',', $values) . ")");
                $ins->execute($params);
                $messages[] = "Admin criado com sucesso: {$email}";
            }

            $done = true;
            if ($deleteAfter) {
                $self = __FILE__;
                @unlink($self);
                $messages[] = "Arquivo fix_admin.php removido automaticamente.";
            }
        }
    }

} catch (Throwable $e) {
    $errors[] = $e->getMessage();
    $defaultEmail = $_POST['email'] ?? 'admin@local.test';
    $defaultName = $_POST['name'] ?? 'Administrador';
    $defaultPassword = $_POST['password'] ?? '123456';
}
?>
<!doctype html>
<html lang="pt-br">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <title>Fix Admin</title>
  <style>
    body{font-family:Arial,Helvetica,sans-serif;background:#0f172a;color:#e2e8f0;padding:24px}
    .box{max-width:760px;margin:0 auto;background:#111827;border:1px solid #334155;border-radius:12px;padding:20px}
    h1{margin-top:0}
    label{display:block;margin-top:12px;font-weight:600}
    input[type=text],input[type=email],input[type=password]{width:100%;padding:10px;border-radius:8px;border:1px solid #475569;background:#0b1220;color:#e2e8f0}
    .btn{margin-top:16px;padding:10px 14px;border:0;border-radius:8px;background:#22c55e;color:#052e16;font-weight:700;cursor:pointer}
    .alert{padding:10px 12px;border-radius:8px;margin:12px 0}
    .ok{background:#052e16;color:#86efac;border:1px solid #166534}
    .err{background:#3f0a0a;color:#fecaca;border:1px solid #7f1d1d}
    code{background:#0b1220;padding:2px 6px;border-radius:6px}
  </style>
</head>
<body>
  <div class="box">
    <h1>Patch: Criar/Atualizar Admin</h1>
    <p>Use este utilitário para corrigir login com "Credenciais inválidas ou conta inativa".</p>

    <?php foreach ($errors as $err): ?>
      <div class="alert err"><?= h($err) ?></div>
    <?php endforeach; ?>

    <?php foreach ($messages as $msg): ?>
      <div class="alert ok"><?= h($msg) ?></div>
    <?php endforeach; ?>

    <?php if (!$done): ?>
      <form method="post">
        <label>Nome</label>
        <input type="text" name="name" value="<?= h($defaultName ?? 'Administrador') ?>" required>

        <label>Email</label>
        <input type="email" name="email" value="<?= h($defaultEmail ?? 'admin@local.test') ?>" required>

        <label>Senha</label>
        <input type="password" name="password" value="<?= h($defaultPassword ?? '123456') ?>" required>

        <label style="display:flex;gap:8px;align-items:center;margin-top:12px;font-weight:400">
          <input type="checkbox" name="delete_after" value="1">
          Apagar automaticamente este arquivo após concluir (recomendado)
        </label>

        <button class="btn" type="submit">Corrigir admin agora</button>
      </form>
    <?php else: ?>
      <p>Login sugerido: <code><?= h($defaultEmail ?? 'admin@local.test') ?></code></p>
      <p>Acesse: <code>/video_saas/public/auth/login.php</code></p>
    <?php endif; ?>

    <hr style="border-color:#334155;margin:18px 0;">
    <p><strong>Segurança:</strong> remova manualmente <code>public/fix_admin.php</code> após uso, se não marcou remoção automática.</p>
  </div>
</body>
</html>
