<?php
declare(strict_types=1);
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../app/helpers.php';
require_once __DIR__ . '/../app/services/PlanGuardService.php';

$videoId = (int)($_GET['video_id'] ?? 0);
$file = trim($_GET['file'] ?? 'master.m3u8');
$token = trim($_GET['token'] ?? '');

if ($videoId <= 0 || $file === '' || !preg_match('/^[a-zA-Z0-9._-]+$/', $file)) {
    http_response_code(400);
    exit('Parâmetros inválidos');
}

$st = db()->prepare("SELECT id, user_id, title, storage_type, hls_master_path, video_path, status FROM videos WHERE id=:id LIMIT 1");
$st->execute(['id' => $videoId]);
$video = $st->fetch();

if (!$video || $video['storage_type'] !== 'local') {
    http_response_code(404);
    exit('Vídeo não encontrado');
}

// autorização: sessão do dono/admin OU token externo válido
$ownerId = (int)$video['user_id'];
$allowed = false;
$auth = auth_user();

if ($auth && ((int)$auth['id'] === $ownerId || ($auth['role'] ?? '') === 'admin')) {
    $allowed = true;
}

if (!$allowed && $token !== '' && strlen($token) === 64) {
    $sql = "SELECT l.*
            FROM external_video_links l
            WHERE l.video_id=:v AND l.token=:t AND l.is_enabled=1
            LIMIT 1";
    $lk = db()->prepare($sql);
    $lk->execute(['v' => $videoId, 't' => $token]);
    $row = $lk->fetch();
    if ($row) {
        if (!empty($row['expires_at']) && strtotime($row['expires_at']) < time()) {
            http_response_code(410);
            exit('Link expirado');
        }
        if (!empty($row['max_views']) && (int)$row['views_count'] > (int)$row['max_views']) {
            http_response_code(410);
            exit('Limite de visualizações atingido');
        }
        $allowed = true;
    }
}

if (!$allowed) {
    http_response_code(403);
    exit('Acesso negado');
}

$can = PlanGuardService::canStreamByOwner($ownerId);
if (!$can['ok']) {
    http_response_code(402);
    exit($can['message'] ?? 'Limite de banda');
}

$dir = storage_path('hls/' . $videoId);
$abs = $dir . '/' . $file;
if (!file_exists($abs) || !is_file($abs)) {
    http_response_code(404);
    exit('Arquivo de streaming não encontrado');
}

$ext = strtolower(pathinfo($file, PATHINFO_EXTENSION));
if ($ext === 'm3u8') {
    $content = file_get_contents($abs);
    if ($content === false) {
        http_response_code(500);
        exit('Falha ao ler playlist');
    }

    // reescreve referências relativas para passar novamente por este endpoint
    $lines = preg_split('/\r\n|\r|\n/', $content);
    $base = url('/hls.php?video_id=' . $videoId . '&file=');
    if ($token !== '') $base .= '&token=' . urlencode($token);

    $rewritten = [];
    foreach ($lines as $ln) {
        $trim = trim($ln);
        if ($trim === '' || str_starts_with($trim, '#')) {
            $rewritten[] = $ln;
            continue;
        }
        $f = basename($trim);
        $url = url('/hls.php?video_id=' . $videoId . '&file=' . urlencode($f));
        if ($token !== '') $url .= '&token=' . urlencode($token);
        $rewritten[] = $url;
    }

    header('Content-Type: application/vnd.apple.mpegurl');
    header('Cache-Control: no-store, private');
    echo implode("\n", $rewritten);
    exit;
}

// segmentos
$size = filesize($abs);
header('Content-Type: video/mp2t');
header('Content-Length: ' . $size);
header('Cache-Control: private, max-age=60');
readfile($abs);

PlanGuardService::registerBandwidth($ownerId, $videoId, (int)$size, 'hls_segment');
