<?php
declare(strict_types=1);

/**
 * One-click installer for PHP/MySQL projects.
 * - Checks required PHP extensions
 * - Creates .env file
 * - Validates DB connection
 * - Imports SQL schema automatically
 *
 * Place this file in /public/setup.php
 */

$projectRoot = realpath(__DIR__ . '/..');
if ($projectRoot === false) {
    http_response_code(500);
    exit('Erro: não foi possível localizar a raiz do projeto.');
}

$lockFile = $projectRoot . DIRECTORY_SEPARATOR . 'storage' . DIRECTORY_SEPARATOR . 'install.lock';
if (!is_dir(dirname($lockFile))) {
    @mkdir(dirname($lockFile), 0775, true);
}
$envPath = $projectRoot . DIRECTORY_SEPARATOR . '.env';
$envExamplePath = $projectRoot . DIRECTORY_SEPARATOR . '.env.example';
$schemaPath = $projectRoot . DIRECTORY_SEPARATOR . 'sql' . DIRECTORY_SEPARATOR . 'schema.sql';

$requiredExtensions = [
    'pdo',
    'pdo_mysql',
    'openssl',
    'curl',
    'mbstring',
    'json',
    'fileinfo',
];

function h(string $v): string { return htmlspecialchars($v, ENT_QUOTES, 'UTF-8'); }

function parseEnvFile(string $path): array {
    $out = [];
    if (!is_file($path)) return $out;
    $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) ?: [];
    foreach ($lines as $line) {
        if (str_starts_with(trim($line), '#')) continue;
        $pos = strpos($line, '=');
        if ($pos === false) continue;
        $k = trim(substr($line, 0, $pos));
        $v = trim(substr($line, $pos + 1));
        $v = trim($v, "\"'");
        $out[$k] = $v;
    }
    return $out;
}

function buildEnv(array $cfg): string {
    $defaults = [
        'APP_NAME' => 'Video SaaS',
        'APP_ENV' => 'production',
        'APP_DEBUG' => 'false',
        'APP_URL' => 'http://localhost/video_saas/public',
        'APP_KEY' => bin2hex(random_bytes(16)),
        'DB_HOST' => '127.0.0.1',
        'DB_PORT' => '3306',
        'DB_DATABASE' => 'video_saas',
        'DB_USERNAME' => 'root',
        'DB_PASSWORD' => '',
        'TIMEZONE' => 'America/Sao_Paulo',
    ];
    $data = array_merge($defaults, $cfg);

    $lines = [];
    foreach ($data as $k => $v) {
        $v = (string)$v;
        if (preg_match('/\s/', $v)) $v = '"' . addslashes($v) . '"';
        $lines[] = $k . '=' . $v;
    }
    return implode(PHP_EOL, $lines) . PHP_EOL;
}



function pdoOptions(): array {
    $opts = [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => true,
    ];
    if (defined('PDO::MYSQL_ATTR_USE_BUFFERED_QUERY')) {
        $opts[PDO::MYSQL_ATTR_USE_BUFFERED_QUERY] = true;
    }
    return $opts;
}

function testDb(string $host, string $port, string $db, string $user, string $pass): array {
    try {
        $dsn = "mysql:host={$host};port={$port};dbname={$db};charset=utf8mb4";
        $pdo = new PDO($dsn, $user, $pass, pdoOptions());
        $stmt = $pdo->query('SELECT 1');
        if ($stmt instanceof PDOStatement) { $stmt->fetchAll(); $stmt->closeCursor(); }
        return [true, 'Conexão com banco OK.'];
    } catch (Throwable $e) {
        return [false, 'Falha na conexão: ' . $e->getMessage()];
    }
}


function splitSqlStatements(string $sql): array {
    $sql = preg_replace('/^ï»¿/', '', $sql); // remove UTF-8 BOM

    $statements = [];
    $current = '';

    $inSingle = false;
    $inDouble = false;
    $inBacktick = false;
    $inLineComment = false;
    $inBlockComment = false;
    $escape = false;

    $len = strlen($sql);
    for ($i = 0; $i < $len; $i++) {
        $ch = $sql[$i];
        $next = $i + 1 < $len ? $sql[$i + 1] : '';

        if ($inLineComment) {
            if ($ch === "
") {
                $inLineComment = false;
                $current .= $ch;
            }
            continue;
        }

        if ($inBlockComment) {
            if ($ch === '*' && $next === '/') {
                $inBlockComment = false;
                $i++; // consume '/'
            }
            continue;
        }

        if (!$inSingle && !$inDouble && !$inBacktick) {
            if ($ch === '-' && $next === '-' ) {
                $prev = $i > 0 ? $sql[$i-1] : "
";
                $next2 = $i + 2 < $len ? $sql[$i+2] : "
";
                // SQL line comment: -- followed by space/control
                if (preg_match('/\s/', $next2) && preg_match('/\s/', $prev)) {
                    $inLineComment = true;
                    $i++; // consume second '-'
                    continue;
                }
            }
            if ($ch === '#') {
                $inLineComment = true;
                continue;
            }
            if ($ch === '/' && $next === '*') {
                $inBlockComment = true;
                $i++; // consume '*'
                continue;
            }
        }

        if (!$inDouble && !$inBacktick && $ch === "'" && !$escape) {
            $inSingle = !$inSingle;
            $current .= $ch;
            continue;
        }
        if (!$inSingle && !$inBacktick && $ch === '"' && !$escape) {
            $inDouble = !$inDouble;
            $current .= $ch;
            continue;
        }
        if (!$inSingle && !$inDouble && $ch === '`' && !$escape) {
            $inBacktick = !$inBacktick;
            $current .= $ch;
            continue;
        }

        if ($ch === ';' && !$inSingle && !$inDouble && !$inBacktick) {
            $stmt = trim($current);
            if ($stmt !== '') {
                $statements[] = $stmt;
            }
            $current = '';
            $escape = false;
            continue;
        }

        $current .= $ch;

        if (($inSingle || $inDouble) && $ch === '\\' && !$escape) {
            $escape = true;
        } else {
            $escape = false;
        }
    }

    $tail = trim($current);
    if ($tail !== '') {
        $statements[] = $tail;
    }

    return $statements;
}

function importSchema(PDO $pdo, string $schemaPath): array {
    if (!is_file($schemaPath)) {
        return [false, "Arquivo de schema não encontrado em: {$schemaPath}"];
    }

    $sql = file_get_contents($schemaPath);
    if ($sql === false || trim($sql) === '') {
        return [false, "schema.sql vazio ou ilegível."];
    }

    // Remove comandos DELIMITER (não necessários para este schema)
    $sql = preg_replace('/^\s*DELIMITER\s+.+$/mi', '', $sql);

    $statements = splitSqlStatements($sql);
    $count = 0;

    try {
        if (defined('PDO::MYSQL_ATTR_USE_BUFFERED_QUERY')) {
            $pdo->setAttribute(PDO::MYSQL_ATTR_USE_BUFFERED_QUERY, true);
        }

        foreach ($statements as $idx => $stmt) {
            $stmt = trim($stmt);
            if ($stmt === '') {
                continue;
            }

            $ok = $pdo->exec($stmt);
            if ($ok === false) {
                $err = $pdo->errorInfo();
                $preview = mb_substr(preg_replace('/\s+/', ' ', $stmt), 0, 180);
                return [false, "Erro ao importar schema no comando #" . ($idx + 1) . ": " . ($err[2] ?? 'erro desconhecido') . " | SQL: {$preview}"];
            }
            $count++;
        }

        return [true, "Schema importado com sucesso. Comandos executados: {$count}"];
    } catch (Throwable $e) {
        return [false, 'Erro ao importar schema: ' . $e->getMessage()];
    }
}


function findFirstExistingColumn(array $columnsMeta, array $candidates): ?string {
    $lowerToReal = [];
    foreach ($columnsMeta as $c) {
        $lowerToReal[strtolower($c['Field'])] = $c['Field'];
    }
    foreach ($candidates as $cand) {
        $k = strtolower($cand);
        if (isset($lowerToReal[$k])) return $lowerToReal[$k];
    }
    return null;
}

function enumOptionsFromType(string $type): array {
    // enum('a','b','c')
    if (!preg_match("/^enum\((.*)\)$/i", trim($type), $m)) return [];
    $raw = $m[1];
    $parts = str_getcsv($raw, ',', "'");
    return array_values(array_filter(array_map('trim', $parts), fn($v) => $v !== ''));
}

function chooseStatusValue(string $colType) {
    $t = strtolower($colType);
    if (str_contains($t, 'int') || str_contains($t, 'bool')) return 1;

    $optsOriginal = enumOptionsFromType($colType);
    $opts = array_map('strtolower', $optsOriginal);
    foreach (['active','ativo','enabled','enable','on'] as $wanted) {
        $idx = array_search($wanted, $opts, true);
        if ($idx !== false) return $optsOriginal[$idx];
    }
    return null;
}

function chooseRoleValue(string $colType): ?string {
    $opts = enumOptionsFromType($colType);
    if (!$opts) return 'admin';
    foreach ($opts as $opt) {
        if (strtolower($opt) === 'admin') return $opt;
    }
    foreach ($opts as $opt) {
        if (str_contains(strtolower($opt), 'admin')) return $opt;
    }
    return $opts[0] ?? null;
}

function detectUserTable(PDO $pdo): ?string {
    $candidates = ['users','admins','clientes','clients'];
    foreach ($candidates as $t) {
        $stmt = $pdo->prepare("SHOW TABLES LIKE :t");
        $stmt->execute(['t' => $t]);
        $exists = (bool)$stmt->fetchColumn();
        $stmt->closeCursor();
        if ($exists) return $t;
    }
    return null;
}

function createOrUpdateAdmin(PDO $pdo, string $email, string $plainPassword, string $defaultName = 'Administrador'): array {
    $table = detectUserTable($pdo);
    if (!$table) {
        return [false, "Aviso: nenhuma tabela de usuários encontrada (users/admins/clientes/clients)."];
    }

    $colsStmt = $pdo->query("SHOW COLUMNS FROM `{$table}`");
    $columnsMeta = $colsStmt->fetchAll(PDO::FETCH_ASSOC);
    $colsStmt->closeCursor();
    if (!$columnsMeta) {
        return [false, "Aviso: não foi possível ler colunas da tabela {$table}."];
    }

    $emailCol  = findFirstExistingColumn($columnsMeta, ['email','mail','login']);
    $passCol   = findFirstExistingColumn($columnsMeta, ['password_hash','password','senha','passwd','pass_hash']);
    $nameCol   = findFirstExistingColumn($columnsMeta, ['name','nome','full_name']);
    $roleCol   = findFirstExistingColumn($columnsMeta, ['role','perfil','user_role']);
    $statusCol = findFirstExistingColumn($columnsMeta, ['status','is_active','active','enabled']);
    $idCol     = findFirstExistingColumn($columnsMeta, ['id','user_id','admin_id']);
    $createdCol= findFirstExistingColumn($columnsMeta, ['created_at','createdon','created']);
    $updatedCol= findFirstExistingColumn($columnsMeta, ['updated_at','updatedon','updated']);

    if (!$emailCol || !$passCol) {
        $available = implode(', ', array_map(fn($c) => $c['Field'], $columnsMeta));
        return [false, "Aviso: tabela {$table} incompatível para admin automático. Colunas atuais: {$available}"];
    }

    $email = strtolower(trim($email));
    $passHash = password_hash($plainPassword, PASSWORD_BCRYPT);

    // map field types
    $typeByField = [];
    foreach ($columnsMeta as $c) {
        $typeByField[strtolower($c['Field'])] = $c['Type'] ?? '';
    }

    $sel = $pdo->prepare("SELECT * FROM `{$table}` WHERE `{$emailCol}` = :email LIMIT 1");
    $sel->execute(['email' => $email]);
    $row = $sel->fetch(PDO::FETCH_ASSOC);
    $sel->closeCursor();

    $roleValue = null;
    if ($roleCol) {
        $roleType = $typeByField[strtolower($roleCol)] ?? '';
        $roleValue = chooseRoleValue($roleType);
    }

    $statusValue = null;
    if ($statusCol) {
        $statusType = $typeByField[strtolower($statusCol)] ?? '';
        $statusValue = chooseStatusValue($statusType);
    }

    if ($row) {
        if (!$idCol || !isset($row[$idCol])) {
            return [false, "Aviso: registro encontrado em {$table}, mas coluna de ID não detectada para update seguro."];
        }

        $sets = ["`{$passCol}` = :pass"];
        $params = ['pass' => $passHash, 'id' => $row[$idCol]];

        if ($roleCol && $roleValue !== null) {
            $sets[] = "`{$roleCol}` = :role";
            $params['role'] = $roleValue;
        }
        if ($statusCol && $statusValue !== null) {
            $sets[] = "`{$statusCol}` = :status";
            $params['status'] = $statusValue;
        }
        if ($updatedCol) {
            $sets[] = "`{$updatedCol}` = NOW()";
        }

        $upSql = "UPDATE `{$table}` SET " . implode(', ', $sets) . " WHERE `{$idCol}` = :id";
        $up = $pdo->prepare($upSql);
        $up->execute($params);

        return [true, "Usuário admin atualizado automaticamente em {$table}: {$email}"];
    }

    $fields = [];
    $values = [];
    $params = [];

    if ($nameCol) {
        $fields[] = "`{$nameCol}`";
        $values[] = ":name";
        $params['name'] = $defaultName;
    }

    $fields[] = "`{$emailCol}`";
    $values[] = ":email";
    $params['email'] = $email;

    $fields[] = "`{$passCol}`";
    $values[] = ":pass";
    $params['pass'] = $passHash;

    if ($roleCol && $roleValue !== null) {
        $fields[] = "`{$roleCol}`";
        $values[] = ":role";
        $params['role'] = $roleValue;
    }
    if ($statusCol && $statusValue !== null) {
        $fields[] = "`{$statusCol}`";
        $values[] = ":status";
        $params['status'] = $statusValue;
    }
    if ($createdCol) {
        $fields[] = "`{$createdCol}`";
        $values[] = "NOW()";
    }
    if ($updatedCol) {
        $fields[] = "`{$updatedCol}`";
        $values[] = "NOW()";
    }

    $inSql = "INSERT INTO `{$table}` (" . implode(', ', $fields) . ") VALUES (" . implode(', ', $values) . ")";
    $in = $pdo->prepare($inSql);
    $in->execute($params);

    return [true, "Usuário admin criado automaticamente em {$table}: {$email}"];
}

$envExisting = parseEnvFile($envPath);
$msg = [];
$errors = [];

$defaults = [
    'app_name' => $envExisting['APP_NAME'] ?? 'Video SaaS',
    'app_url' => $envExisting['APP_URL'] ?? 'http://localhost/video_saas/public',
    'db_host' => $envExisting['DB_HOST'] ?? '127.0.0.1',
    'db_port' => $envExisting['DB_PORT'] ?? '3306',
    'db_name' => $envExisting['DB_DATABASE'] ?? 'video_saas',
    'db_user' => $envExisting['DB_USERNAME'] ?? 'root',
    'db_pass' => $envExisting['DB_PASSWORD'] ?? '',
    'admin_email' => 'admin@local.test',
    'admin_password' => '123456',
    'import_schema' => '1',
    'lock_after_install' => '1',
];

if (is_file($lockFile)) {
    $msg[] = "Instalação já concluída (install.lock encontrado). Exclua storage/install.lock para reinstalar.";
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && !is_file($lockFile)) {
    foreach ($defaults as $k => $v) {
        $defaults[$k] = $_POST[$k] ?? $v;
    }

    // Extension checks
    foreach ($requiredExtensions as $ext) {
        if (!extension_loaded($ext)) {
            $errors[] = "Extensão PHP ausente: {$ext}";
        }
    }

    if (!filter_var($defaults['app_url'], FILTER_VALIDATE_URL)) {
        $errors[] = "APP_URL inválida.";
    }

    if (!preg_match('/^\d+$/', (string)$defaults['db_port'])) {
        $errors[] = "Porta do banco inválida.";
    }

    if (empty($errors)) {
        // test DB first
        [$okDb, $dbMsg] = testDb(
            (string)$defaults['db_host'],
            (string)$defaults['db_port'],
            (string)$defaults['db_name'],
            (string)$defaults['db_user'],
            (string)$defaults['db_pass']
        );

        if (!$okDb) {
            $errors[] = $dbMsg;
        } else {
            $msg[] = $dbMsg;

            // write .env
            $envData = [
                'APP_NAME' => $defaults['app_name'],
                'APP_ENV' => 'production',
                'APP_DEBUG' => 'false',
                'APP_URL' => $defaults['app_url'],
                'APP_KEY' => bin2hex(random_bytes(16)),
                'DB_HOST' => $defaults['db_host'],
                'DB_PORT' => $defaults['db_port'],
                'DB_DATABASE' => $defaults['db_name'],
                'DB_USERNAME' => $defaults['db_user'],
                'DB_PASSWORD' => $defaults['db_pass'],
                'TIMEZONE' => 'America/Sao_Paulo',
            ];
            $envString = buildEnv($envData);
            $w = @file_put_contents($envPath, $envString);
            if ($w === false) {
                $errors[] = "Não foi possível gravar o arquivo .env em {$envPath}";
            } else {
                $msg[] = ".env criado com sucesso.";
            }
            // import schema
            if (empty($errors) && $defaults['import_schema'] === '1') {
                try {
                    $dsn = "mysql:host={$defaults['db_host']};port={$defaults['db_port']};dbname={$defaults['db_name']};charset=utf8mb4";
                    $pdo = new PDO($dsn, (string)$defaults['db_user'], (string)$defaults['db_pass'], pdoOptions());

                    [$okSchema, $schemaMsg] = importSchema($pdo, $schemaPath);
                    if (!$okSchema) {
                        $errors[] = $schemaMsg;
                    } else {
                        $msg[] = $schemaMsg;
                    }

                    // create/update admin user automatically (schema-aware)
                    if ($okSchema && !empty($defaults['admin_email']) && !empty($defaults['admin_password'])) {
                        try {
                            [$okAdmin, $adminMessage] = createOrUpdateAdmin(
                                $pdo,
                                (string)$defaults['admin_email'],
                                (string)$defaults['admin_password'],
                                'Administrador'
                            );
                            $msg[] = $adminMessage;
                        } catch (Throwable $e) {
                            $msg[] = "Aviso: não foi possível criar/atualizar admin automaticamente ({$e->getMessage()}).";
                        }
                    }

                } catch (Throwable $e) {
                    $errors[] = "Erro ao conectar/importar schema: " . $e->getMessage();
                }
            }

            if (empty($errors) && $defaults['lock_after_install'] === '1') {

                @file_put_contents($lockFile, date('c') . PHP_EOL);
                $msg[] = "Instalação finalizada e bloqueada com install.lock";
            }
        }
    }
}

$extStatus = [];
foreach ($requiredExtensions as $ext) {
    $extStatus[$ext] = extension_loaded($ext);
}
?>
<!doctype html>
<html lang="pt-br">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <title>Setup 1-Click - Video SaaS</title>
  <style>
    :root{
      --bg:#0b1220; --card:#111a2e; --muted:#9db0d0; --txt:#e9f1ff; --ok:#28d17c; --err:#ff5c7c; --pri:#49a6ff;
      --border:rgba(255,255,255,.08);
    }
    *{box-sizing:border-box}
    body{
      margin:0; font-family:Inter,Segoe UI,Arial,sans-serif; color:var(--txt);
      background: radial-gradient(1200px 500px at 0% -10%, #1f3a8a66, transparent),
                  radial-gradient(1000px 400px at 100% 110%, #0ea5e966, transparent),
                  linear-gradient(135deg,#060b16,#0b1220);
      min-height:100vh; padding:28px;
    }
    .wrap{max-width:980px;margin:0 auto}
    .card{
      background: linear-gradient(180deg, rgba(255,255,255,.05), rgba(255,255,255,.02));
      border:1px solid var(--border); border-radius:16px; padding:20px; margin-bottom:16px;
      backdrop-filter: blur(7px);
      box-shadow: 0 10px 30px rgba(0,0,0,.25);
    }
    h1{margin:0 0 4px;font-size:28px}
    p{margin:6px 0;color:var(--muted)}
    .grid{display:grid;grid-template-columns:repeat(2,minmax(0,1fr));gap:12px}
    label{display:block;font-size:12px;color:#b9c9e6;margin-bottom:6px}
    input{width:100%;padding:10px 12px;border-radius:10px;border:1px solid var(--border);background:#0b1426;color:#eaf2ff}
    input:focus{outline:none;border-color:#3b82f6;box-shadow:0 0 0 3px rgba(59,130,246,.2)}
    .full{grid-column:1/-1}
    .btn{
      background:linear-gradient(90deg,#2563eb,#06b6d4);color:white;border:none;border-radius:12px;padding:12px 16px;
      font-weight:700;cursor:pointer; margin-top:8px;
    }
    .muted{font-size:13px;color:var(--muted)}
    .badge{display:inline-block;padding:4px 10px;border-radius:999px;font-size:12px;border:1px solid var(--border)}
    .ok{color:var(--ok)} .err{color:var(--err)}
    .list{margin:8px 0 0 0;padding-left:18px}
    .status-grid{display:grid;grid-template-columns:repeat(4,minmax(0,1fr));gap:10px}
    .status{padding:10px;border:1px solid var(--border);border-radius:10px;background:#0b1426}
    .foot{font-size:12px;color:#95a8cb}
    @media (max-width: 780px){
      .grid,.status-grid{grid-template-columns:1fr}
    }
  </style>
</head>
<body>
<div class="wrap">
  <div class="card">
    <h1>Instalador 1-Click</h1>
    <p>Cria <code>.env</code>, testa extensões PHP e valida banco MySQL automaticamente.</p>
    <p class="muted">Projeto: <code><?= h((string)$projectRoot) ?></code></p>
    <div>
      <span class="badge">PHP <?= h(PHP_VERSION) ?></span>
      <span class="badge">Servidor <?= h($_SERVER['SERVER_SOFTWARE'] ?? 'N/A') ?></span>
      <span class="badge"><?= is_file($lockFile) ? 'Install: bloqueado' : 'Install: liberado' ?></span>
    </div>
  </div>

  <div class="card">
    <h3 style="margin-top:0;">Extensões necessárias</h3>
    <div class="status-grid">
      <?php foreach ($extStatus as $ext => $ok): ?>
        <div class="status">
          <strong><?= h($ext) ?></strong><br>
          <span class="<?= $ok ? 'ok' : 'err' ?>"><?= $ok ? 'OK' : 'Faltando' ?></span>
        </div>
      <?php endforeach; ?>
    </div>
  </div>

  <?php if (!empty($msg)): ?>
    <div class="card">
      <h3 style="margin-top:0;" class="ok">Mensagens</h3>
      <ul class="list"><?php foreach ($msg as $m): ?><li><?= h($m) ?></li><?php endforeach; ?></ul>
    </div>
  <?php endif; ?>

  <?php if (!empty($errors)): ?>
    <div class="card">
      <h3 style="margin-top:0;" class="err">Erros</h3>
      <ul class="list"><?php foreach ($errors as $e): ?><li><?= h($e) ?></li><?php endforeach; ?></ul>
    </div>
  <?php endif; ?>

  <div class="card">
    <form method="post">
      <div class="grid">
        <div>
          <label>APP_NAME</label>
          <input name="app_name" value="<?= h((string)$defaults['app_name']) ?>">
        </div>
        <div>
          <label>APP_URL</label>
          <input name="app_url" value="<?= h((string)$defaults['app_url']) ?>">
        </div>

        <div>
          <label>DB_HOST</label>
          <input name="db_host" value="<?= h((string)$defaults['db_host']) ?>">
        </div>
        <div>
          <label>DB_PORT</label>
          <input name="db_port" value="<?= h((string)$defaults['db_port']) ?>">
        </div>
        <div>
          <label>DB_DATABASE</label>
          <input name="db_name" value="<?= h((string)$defaults['db_name']) ?>">
        </div>
        <div>
          <label>DB_USERNAME</label>
          <input name="db_user" value="<?= h((string)$defaults['db_user']) ?>">
        </div>
        <div class="full">
          <label>DB_PASSWORD</label>
          <input type="password" name="db_pass" value="<?= h((string)$defaults['db_pass']) ?>">
        </div>

        <div>
          <label>ADMIN_EMAIL</label>
          <input type="email" name="admin_email" value="<?= h((string)$defaults['admin_email']) ?>">
        </div>
        <div>
          <label>ADMIN_PASSWORD</label>
          <input type="text" name="admin_password" value="<?= h((string)$defaults['admin_password']) ?>">
        </div>

        <div>
          <label><input type="checkbox" name="import_schema" value="1" <?= ($defaults['import_schema']==='1'?'checked':'') ?>> Importar schema.sql</label>
        </div>
        <div>
          <label><input type="checkbox" name="lock_after_install" value="1" <?= ($defaults['lock_after_install']==='1'?'checked':'') ?>> Criar install.lock ao final</label>
        </div>
      </div>

      <button class="btn" type="submit" <?= is_file($lockFile) ? 'disabled' : '' ?>>Executar instalação</button>
      <p class="foot">Depois de concluir, remova <code>public/setup.php</code> em produção.</p>
    </form>
  </div>
</div>
</body>
</html>
