<?php
declare(strict_types=1);
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../app/helpers.php';
require_once __DIR__ . '/../app/services/PlanGuardService.php';
require_once __DIR__ . '/../app/services/VideoPipelineService.php';
require_auth();

$u = auth_user();
$uid = (int)$u['id'];

if (is_post()) {
    if (!csrf_check($_POST['_csrf'] ?? null)) {
        flash('error', 'CSRF inválido.');
        redirect('/video_upload.php');
    }

    $title = trim($_POST['title'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $storageType = $_POST['storage_type'] ?? 'local';
    $externalUrl = trim($_POST['external_url'] ?? '');
    $visibility = ($_POST['visibility'] ?? 'private') === 'public' ? 'public' : 'private';

    if ($title === '') {
        flash('error', 'Informe um título.');
        redirect('/video_upload.php');
    }

    if ($storageType === 'external') {
        if (!filter_var($externalUrl, FILTER_VALIDATE_URL)) {
            flash('error', 'URL externa inválida.');
            redirect('/video_upload.php');
        }
        db()->prepare("INSERT INTO videos (user_id,title,description,storage_type,external_url,visibility,status) VALUES (:u,:t,:d,'external',:url,:vis,'ready')")
            ->execute(['u'=>$uid,'t'=>$title,'d'=>$description,'url'=>$externalUrl,'vis'=>$visibility]);
        flash('success', 'Vídeo externo cadastrado.');
        redirect('/videos.php');
    }

    if (!isset($_FILES['video']) || !is_uploaded_file($_FILES['video']['tmp_name'])) {
        flash('error', 'Selecione um arquivo de vídeo.');
        redirect('/video_upload.php');
    }

    $fileBytes = (int)$_FILES['video']['size'];
    $gate = PlanGuardService::canUpload($uid, $fileBytes);
    if (!$gate['ok']) {
        $msg = $gate['message'] ?? 'Seu plano não permite este upload.';
        if (!empty($gate['plan'])) {
            $plan = $gate['plan'];
            $usage = $gate['usage'] ?? [];
            $msg .= ' | Plano: ' . $plan['name']
                . ' | Uso atual: ' . human_bytes((int)($usage['storage_used_bytes'] ?? 0))
                . ' de ' . ((int)$plan['max_storage_gb']) . ' GB';
        }
        flash('error', $msg . ' Ajuste seu plano em /plans.php');
        redirect('/video_upload.php');
    }

    $allowed = ['video/mp4','video/webm','video/quicktime','video/x-matroska','application/octet-stream'];
    $mime = mime_content_type($_FILES['video']['tmp_name']) ?: '';
    if (!in_array($mime, $allowed, true)) {
        flash('error', 'Formato não permitido. Use MP4/WEBM/MOV/MKV.');
        redirect('/video_upload.php');
    }

    $sizeMb = bytes_to_mb($fileBytes);
    if ($sizeMb > MAX_UPLOAD_MB) {
        flash('error', 'Arquivo excede limite de ' . MAX_UPLOAD_MB . ' MB.');
        redirect('/video_upload.php');
    }

    $ext = strtolower(pathinfo($_FILES['video']['name'], PATHINFO_EXTENSION) ?: 'mp4');
    $name = 'v_' . $uid . '_' . bin2hex(random_bytes(8)) . '.' . $ext;
    $uploadDir = ensure_upload_dir();
    $dest = $uploadDir . '/' . $name;

    if (!move_uploaded_file($_FILES['video']['tmp_name'], $dest)) {
        flash('error', 'Falha ao salvar o vídeo.');
        redirect('/video_upload.php');
    }

    $relPath = 'uploads/' . $name;

    $ins = db()->prepare("
      INSERT INTO videos
      (user_id,title,description,storage_type,video_path,visibility,status,file_size_bytes,original_filename,mime_type)
      VALUES
      (:u,:t,:d,'local',:vp,:vis,'processing',:fs,:of,:mm)
    ");
    $ins->execute([
        'u'=>$uid,
        't'=>$title,
        'd'=>$description,
        'vp'=>$relPath,
        'vis'=>$visibility,
        'fs'=>$fileBytes,
        'of'=>substr((string)($_FILES['video']['name'] ?? ''), 0, 190),
        'mm'=>substr($mime, 0, 120),
    ]);
    $videoId = (int)db()->lastInsertId();

    VideoPipelineService::enqueueTranscode($videoId);

    flash('success', 'Upload concluído. Vídeo enviado para processamento HLS em fila.');
    redirect('/videos.php');
}

$pageTitle = 'Enviar Vídeo';
include __DIR__ . '/partials/head.php';
include __DIR__ . '/partials/navbar.php';
include __DIR__ . '/partials/sidebar.php';
?>
<div class="content-wrapper">
  <div class="content-header"><div class="container-fluid"><h1>Enviar vídeo</h1></div></div>
  <section class="content"><div class="container-fluid">
    <?php include __DIR__ . '/partials/flash.php'; ?>
    <div class="card glass-card">
      <div class="card-body">
        <form method="post" enctype="multipart/form-data">
          <input type="hidden" name="_csrf" value="<?= e(csrf_token()) ?>">
          <div class="row g-3">
            <div class="col-md-6">
              <label class="form-label">Título</label>
              <input class="form-control bg-transparent text-light" type="text" name="title" required>
            </div>
            <div class="col-md-6">
              <label class="form-label">Visibilidade</label>
              <select class="form-select bg-transparent text-light" name="visibility">
                <option value="private">Privado</option>
                <option value="public">Público</option>
              </select>
            </div>
            <div class="col-12">
              <label class="form-label">Descrição</label>
              <textarea class="form-control bg-transparent text-light" name="description" rows="3"></textarea>
            </div>
            <div class="col-md-4">
              <label class="form-label">Origem</label>
              <select class="form-select bg-transparent text-light" name="storage_type" id="storageType">
                <option value="local">Upload local (HLS)</option>
                <option value="external">Link externo</option>
              </select>
            </div>
            <div class="col-md-8" id="localUpload">
              <label class="form-label">Arquivo de vídeo</label>
              <input class="form-control bg-transparent text-light" type="file" name="video" accept="video/*">
              <small class="text-secondary">Após upload, o vídeo entra em fila para transcodificação (360p/720p).</small>
            </div>
            <div class="col-md-8 d-none" id="externalUrlWrap">
              <label class="form-label">URL externa (YouTube/Vimeo/CDN)</label>
              <input class="form-control bg-transparent text-light" type="url" name="external_url" placeholder="https://...">
            </div>
          </div>
          <div class="mt-3">
            <button class="btn btn-neon">Salvar vídeo</button>
          </div>
        </form>
      </div>
    </div>
  </div></section>
</div>
<script>
document.addEventListener('DOMContentLoaded', () => {
  const storage = document.getElementById('storageType');
  const local = document.getElementById('localUpload');
  const external = document.getElementById('externalUrlWrap');
  const toggle = () => {
    if (storage.value === 'external') {
      local.classList.add('d-none');
      external.classList.remove('d-none');
    } else {
      local.classList.remove('d-none');
      external.classList.add('d-none');
    }
  };
  storage.addEventListener('change', toggle);
  toggle();
});
</script>
<?php include __DIR__ . '/partials/footer.php'; ?>
